package ru.yandex.chemodan.app.dataapi.apps.profile.yatickets;

import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;

import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.profile.yatickets.model.OrderTest;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.misc.test.Assert;

/**
 * @author Denis Bakharev
 */
@ActivateDataApiEmbeddedPg
public class OrdersManagerTest extends DataApiTestSupport {

    public static final String ORDER_ID = "oid";
    private DataApiUserId uid;

    @Autowired
    OrdersManager ordersManager;

    @Value("${dataapi.apps.profile.yatickets.order.deletion-interval}")
    Duration deleteInterval;

    @Before
    public void before() {
        uid = createRandomCleanUser();
    }

    @Test
    public void createOrder_NewOrder_MustScheduleDeletion() {
        ordersManager.insertEntity(uid, OrderTest.getSampleOrder(ORDER_ID));
        assertDeletionTaskSchedule();
    }

    @Test
    public void setOrder_NewOrOldOrder_MustScheduleDeletion() {
        ordersManager.setEntity(uid, OrderTest.getSampleOrder(ORDER_ID));
        assertDeletionTaskSchedule();
    }

    @Test
    public void deleteOutdatedOrder_OrderNotExists_DoNothing() {
        boolean isDeleted = ordersManager.deleteOutdatedOrder(uid, "unknown");
        Assert.isFalse(isDeleted);
    }

    @Test
    public void deleteOutdatedOrder_InvokedBeforeExpectedDeletionDate_ScheduleNewTaskWithCorrectDate() {
        ordersManager.insertEntity(uid, OrderTest.getSampleOrder(ORDER_ID, DateTime.now()));
        bazingaStub.tasksWithParams.clear();

        boolean isDeleted = ordersManager.deleteOutdatedOrder(uid, ORDER_ID);
        Assert.isFalse(isDeleted);
        assertDeletionTaskSchedule();
    }

    @Test
    public void deleteOutdatedOrder_AfterDeletionDate_DeleteOrder() {
        DateTime dateInOrder = DateTime.now().minus(deleteInterval).minusMinutes(1);
        ordersManager.insertEntity(uid, OrderTest.getSampleOrder(ORDER_ID, dateInOrder));
        bazingaStub.tasksWithParams.clear();

        boolean isDeleted = ordersManager.deleteOutdatedOrder(uid, ORDER_ID);
        Assert.isTrue(isDeleted);

        Assert.assertEmpty(bazingaStub.tasksWithParams.filterByType(DeleteOrderTask.class));
    }

    private void assertDeletionTaskSchedule() {
        DeleteOrderTask task = bazingaStub.tasksWithParams.filterByType(DeleteOrderTask.class).single();

        Assert.equals(uid, task.getParametersTyped().uid);
        Assert.equals(ORDER_ID, task.getParametersTyped().orderId);
    }

}
