package ru.yandex.chemodan.app.dataapi.apps.settings;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function1B;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContext;
import ru.yandex.chemodan.app.dataapi.api.db.filter.DatabasesFilter;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefs;
import ru.yandex.misc.lang.Validate;

/**
 * @author dbrylev
 */
public class AppSettings {
    private static final Function1B<AppDatabaseSettings> isUseCacheF = s -> s.useCache;
    public final Option<String> appName;
    private final ListF<AppDatabaseSettings> dbSettings;

    AppSettings(Option<String> appName, ListF<AppDatabaseSettings> dbSettings) {
        this.appName = appName;
        this.dbSettings = dbSettings;
    }

    public DatabasesFilter getDatabasesFilter() {
        return isUseCacheByDefault()
                ? getContext().toDbsFilter().excludeDbRefs(getNoCacheDatabaseRefs())
                : getDatabaseRefs().toDbsFilter();
    }

    private boolean isUseCacheByDefault() {
        return defaultSettingsO().exists(isUseCacheF);
    }

    public boolean isEmpty() {
        return dbSettings.isEmpty();
    }

    public AppDatabaseSettings getSettingsByDbId(Option<String> dbId) {
        return getByDbId(dbId)
                .orElse(this::defaultSettingsO)
                .getOrElse(AppDatabaseSettings.DEFAULT);
    }

    AppSettings withoutDatabaseSettings(Option<String> databaseId) {
        return new AppSettings(appName, dbSettings.filterNot(s -> s.databaseId.equals(databaseId)));
    }

    AppSettings withDatabaseSettings(AppDatabaseSettings settings) {
        Validate.equals(appName, settings.appName);
        return new AppSettings(appName, withoutDatabaseSettings(settings.databaseId).dbSettings.plus1(settings));
    }

    ListF<AppDatabaseSettings> all() {
        return dbSettings;
    }

    AppDatabaseSettings getSettings(DatabaseRefSource dbRefSrc) {
        return getSettingsO(dbRefSrc)
                .orElse(this::defaultSettingsO)
                .getOrElse(AppDatabaseSettings.DEFAULT);
    }

    private Option<AppDatabaseSettings> getSettingsO(DatabaseRefSource dbRefSrc) {
        return get(Option.of(dbRefSrc));
    }

    private Option<AppDatabaseSettings> defaultSettingsO() {
        return get(Option.empty());
    }

    private Option<AppDatabaseSettings> get(Option<DatabaseRefSource> dbRefSrc) {
        return dbSettings.find(settings -> settings.isFor(dbRefSrc));
    }

    private Option<AppDatabaseSettings> getByDbId(Option<String> dbId) {
        return dbSettings.find(settings -> settings.isForDbId(dbId));
    }

    private DatabaseContext getContext() {
        return DatabaseContext.cons(appName);
    }

    private DatabaseRefs getNoCacheDatabaseRefs() {
        return new DatabaseRefs(appName, getNoCacheDatabaseIds());
    }

    private ListF<String> getNoCacheDatabaseIds() {
        return dbSettings.filterNot(isUseCacheF)
                .filterMap(db -> db.databaseId);
    }

    private DatabaseRefs getDatabaseRefs() {
        ListF<String> databaseIds = dbSettings.filterMap(db -> db.databaseId);
        return new DatabaseRefs(appName, databaseIds);
    }
}
