package ru.yandex.chemodan.app.dataapi.apps.settings;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContextSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefSource;
import ru.yandex.chemodan.app.dataapi.utils.DataApiZkRegistry;
import ru.yandex.chemodan.util.ping.PingerChecker;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.primitives.registry.ZkPathEscaping;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.lang.Check;

/**
 * @author Denis Bakharev
 */
public class AppSettingsRegistry extends DataApiZkRegistry<AppDatabaseSettings> implements PingerChecker {
    private final DataSize defaultDatabaseSizeLimit;
    private final int defaultDatabasesCountLimit;

    private volatile MapF<Option<String>, AppSettings> settingsMap;

    public AppSettingsRegistry(ZkPath zkPath, DataSize defaultDatabaseSizeLimit, int defaultDatabasesCountLimit) {
        super(zkPath, AppDatabaseSettings.class, s -> getZkPath(s.appName, s.databaseId), ZkPathEscaping.asIs);
        this.defaultDatabaseSizeLimit = defaultDatabaseSizeLimit;
        this.defaultDatabasesCountLimit = defaultDatabasesCountLimit;
    }

    private AppSettings getSettings(DatabaseContextSource ctxSrc) {
        return getSettings(ctxSrc.appNameO());
    }

    public AppDatabaseSettings getDatabaseSettings(DatabaseRefSource dbRefSrc) {
        return getSettings(dbRefSrc)
                .getSettings(dbRefSrc)
                .withDefaultDatabaseSizeLimit(defaultDatabaseSizeLimit);
    }

    public AppSettings getSettings(Option<String> app) {
        Check.notNull(settingsMap, "Not initialized yet");
        return settingsMap.getO(app).getOrElse(new AppSettings(app, Cf.list()));
    }

    public void setSettings(AppDatabaseSettings settings) {
        put(settings);
    }

    public void deleteTypeSettings(Option<String> appName, Option<String> databaseId) {
        remove(getZkPath(appName, databaseId));
    }

    private static String getZkPath(Option<String> appName, Option<String> databaseId) {
        String app = UrlUtils.urlEncode(appName.getOrElse(".global"));
        String db = UrlUtils.urlEncode(databaseId.getOrElse(".any"));

        return app + "+" + db;
    }

    @Override
    protected void handleNewState(CollectionF<AppDatabaseSettings> allElements) {
        settingsMap = allElements.groupBy(e -> e.appName)
                .mapValues(es -> new AppSettings(es.first().appName, es));
    }

    public int getDefaultDatabasesCountLimit() {
        return defaultDatabasesCountLimit;
    }

    @Override
    public boolean isActive() {
        return settingsMap != null;
    }
}
