package ru.yandex.chemodan.app.dataapi.core;

import java.util.function.Consumer;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function0;
import ru.yandex.misc.monica.annotation.GroupByDefault;
import ru.yandex.misc.monica.annotation.MonicaMetric;
import ru.yandex.misc.monica.annotation.MonicaStaticRegistry;
import ru.yandex.misc.monica.core.blocks.InstrumentMap;
import ru.yandex.misc.monica.core.blocks.Profiler;
import ru.yandex.misc.monica.core.blocks.Statistic;
import ru.yandex.misc.monica.core.name.MetricGroupName;
import ru.yandex.misc.monica.core.name.MetricName;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public final class DataApiStats {
    private static final MetricGroupName METRIC_GROUP_NAME =
            new MetricGroupName("dataapi", new MetricName("dataapi", "manager"), "Database manager");

    @MonicaMetric
    @GroupByDefault
    public static final Statistic changesInDelta = new Statistic();

    @MonicaMetric
    @GroupByDefault
    public static final InstrumentMap listenerInvocations = new InstrumentMap();

    @MonicaMetric
    @GroupByDefault
    private static final Profiler applyDeltaProfiler = new Profiler();

    private static class Profiling {
        private static final ThreadLocal<Option<Profiler.Session>> sessionThreadLocal =
                ThreadLocal.withInitial(Option::empty);

        static <T> T executeWithProfiling(Function0<T> op) {
            try {
                startProfiling(false);
                return op.apply();
            } finally {
                stopProfilingIfRunning();
            }
        }

        static void step(String name) {
            withThreadLocalSessionO(session -> session.step(name));
        }

        private static void startProfiling(boolean stopCurrent) {
            if (stopCurrent && isRunning()) {
                stopProfilingIfRunning();
            }

            if (isRunning()) {
                return;
            }

            sessionThreadLocal.set(Option.of(DataApiStats.applyDeltaProfiler.createAndStart()));
        }

        private static void stopProfilingIfRunning() {
            withThreadLocalSessionO(Profiler.Session::end);
        }

        private static void withThreadLocalSessionO(Consumer<Profiler.Session> consumer) {
            if (!isRunning()) {
                return;
            }
            consumer.accept(sessionThreadLocal.get().get());
        }

        private static boolean isRunning() {
            return sessionThreadLocal.get().isPresent();
        }
    }

    public static <T> T executeWithProfiling(Function0<T> op) {
        return Profiling.executeWithProfiling(op);
    }

    public static void step(String name) {
        Profiling.step(name);
    }

    static {
        MonicaStaticRegistry.register(DataApiStats.class, METRIC_GROUP_NAME);
    }

    private DataApiStats() {}
}
