package ru.yandex.chemodan.app.dataapi.core;

import java.lang.reflect.Proxy;
import java.util.concurrent.ExecutorService;

import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Primary;

import ru.yandex.chemodan.app.dataapi.api.DatabaseChangedEventAsyncHandler;
import ru.yandex.chemodan.app.dataapi.api.DatabaseChangedEventHandler;
import ru.yandex.chemodan.app.dataapi.api.datasource.SpecificDataSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltasJdbcDao;
import ru.yandex.chemodan.app.dataapi.apps.CompositeApplicationManager;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DataRecordsJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DeletedDatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.support.ShardedTransactionManager;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.datasources.BasicDataSourceRegistry;
import ru.yandex.chemodan.app.dataapi.core.datasources.DataSourceTypeRegistry;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.datasources.migration.DsMigration;
import ru.yandex.chemodan.app.dataapi.core.datasources.migration.DsMigrationDataSourceRegistry;
import ru.yandex.chemodan.app.dataapi.core.datasources.migration.DsMigrationManager;
import ru.yandex.chemodan.app.dataapi.core.limiter.DatabaseLimiter;
import ru.yandex.chemodan.app.dataapi.core.limiter.access.DataApiAccessRateLimitContextConfiguration;
import ru.yandex.chemodan.app.dataapi.core.limiter.access.DataApiManagerLimitingInvocationHandler;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManagerImpl;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReferenceJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReferenceManager;
import ru.yandex.chemodan.app.dataapi.utils.PriorityExecutorService;
import ru.yandex.chemodan.app.dataapi.utils.elliptics.EllipticsHelper;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;
import ru.yandex.misc.spring.ApplicationContextUtils;

/**
 * @author tolmalev
 */
@Import({
        DataApiAccessRateLimitContextConfiguration.class
})
@Configuration
public class DatabasesContextConfiguration {
    public static final String DATABASE_MANAGER_EXECUTOR_SERVICE = "databaseManagerExecutorService";

    private static final Logger logger = LoggerFactory.getLogger(DatabasesContextConfiguration.class);

    @Bean
    public DatabaseLimiter databaseLimiter(
            DatabasesJdbcDao databasesDao,
            AppSettingsRegistry appSettingsRegistry)
    {
        return new DatabaseLimiter(databasesDao, appSettingsRegistry);
    }

    @Bean
    public DiskDataSource nativeDataApiManager(
            DatabasesJdbcDao databasesJdbcDao,
            DeletedDatabasesJdbcDao deletedDatabasesJdbcDao,
            DeltasJdbcDao deltasJdbcDao,
            ShardedTransactionManager transactionManager,
            DataRecordsJdbcDao protobufRecordsDao,
            CompositeApplicationManager compositeApplicationManager,
            UserMetaManager userMetaManager)
    {
        return new DiskDataSource(
                databasesJdbcDao,
                deletedDatabasesJdbcDao,
                deltasJdbcDao,
                transactionManager,
                protobufRecordsDao,
                compositeApplicationManager,
                userMetaManager);
    }

    @Bean
    public DsMigrationManager dataSourceMigrationManager(
            UserMetaManager userMetaManager,
            BasicDataSourceRegistry dsRegistry,
            DataSourceTypeRegistry dsTypeRegistry,
            ApplicationContext applicationContext) {
        return new DsMigrationManager(userMetaManager, dsTypeRegistry, dsRegistry,
                ApplicationContextUtils.beansOfType(applicationContext, DsMigration.class));
    }

    @Bean
    public DataSourceTypeRegistry dataSourceTypeRegistry(@Qualifier("zkRoot") ZkPath zkRoot, ZkManager zkManager) {
        DataSourceTypeRegistry registry = new DataSourceTypeRegistry(zkRoot.child("data_source_types"));
        registry.addTo(zkManager);
        return registry;
    }

    @Bean
    public BasicDataSourceRegistry basicDataSourceRegistry(
            DataSourceTypeRegistry typeRegistry,
            DiskDataSource diskDataSource,
            ApplicationContext applicationContext) {
        return new BasicDataSourceRegistry(
                typeRegistry,
                diskDataSource,
                ApplicationContextUtils.beansOfType(applicationContext, SpecificDataSource.class)
        );
    }

    @Bean
    @Primary
    public DsMigrationDataSourceRegistry migrationDataSourceRegistry(
            DatabasesJdbcDao databasesJdbcDao,
            UserMetaManager userMetaManager,
            DataSourceTypeRegistry dsTypeRegistry,
            BasicDataSourceRegistry dsRegistry)
    {
        return new DsMigrationDataSourceRegistry(databasesJdbcDao, userMetaManager, dsTypeRegistry, dsRegistry);
    }

    @Bean
    public DataApiManagerImpl dataApiManagerImpl(
            DsMigrationDataSourceRegistry dataSourceRegistry,
            DatabaseLimiter databaseLimiter,
            ExternalDatabasesRegistry externalDatabasesRegistry,
            CompositeApplicationManager appManager,
            AppSettingsRegistry appSettingsRegistry,
            MdsSnapshotReferenceManager mdsSnapshotReferenceManager,
            @Qualifier(DATABASE_MANAGER_EXECUTOR_SERVICE)
            ExecutorService databaseManagerExecutorService,
            ApplicationContext applicationContext)
    {
        return new DataApiManagerImpl(
                dataSourceRegistry,
                databaseLimiter,
                externalDatabasesRegistry,
                appManager,
                appSettingsRegistry,
                mdsSnapshotReferenceManager,
                ApplicationContextUtils.beansOfType(applicationContext, DatabaseChangedEventHandler.class)
                        .sortedBy(DatabaseChangedEventHandler::getOrder),
                ApplicationContextUtils.beansOfType(applicationContext, DatabaseChangedEventAsyncHandler.class),
                databaseManagerExecutorService
        );
    }

    @Bean
    @Primary
    public DataApiManager dataApiManager(DataApiManagerLimitingInvocationHandler handler) {
        return (DataApiManager) Proxy.newProxyInstance(
                DataApiManager.class.getClassLoader(),
                new Class[] { DataApiManager.class },
                handler
        );
    }

    @Bean
    public ExecutorService databaseManagerExecutorService(@Value("${dataapi.priority.pool.slow:-50}") int poolSlow,
            @Value("${dataapi.priority.pool.fast:-100}") int poolFast,
            @Value("${dataapi.priority.pool.max:-200}") int mapPoolSize,
            @Value("${dataapi.priority.pool.keep.minutes:-5}") int keepAliveMinutes)
    {
        return new PriorityExecutorService(poolSlow, poolFast, mapPoolSize, keepAliveMinutes);
    }

    @Bean
    public MdsSnapshotReferenceManager mdsSnapshotReferenceManager(
            MdsSnapshotReferenceJdbcDao mdsSnapshotReferenceJdbcDao,
            @Value("${dataapi.snapshot.deletion-interval}")
            Duration snapshotReferenceDeletionInterval,
            @Value("${dataapi.snapshot.partition-size}")
            int partitionSize,
            EllipticsHelper ellipticsHelper,
            ShardedTransactionManager transactionManager)
    {
        return new MdsSnapshotReferenceManager(
                mdsSnapshotReferenceJdbcDao,
                snapshotReferenceDeletionInterval,
                ellipticsHelper,
                transactionManager, partitionSize);
    }
}
