package ru.yandex.chemodan.app.dataapi.core.dao.data;

import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.dao.IncorrectResultSizeDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContext;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DatabaseCondition;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.filter.DatabasesFilter;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandles;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.deltas.cleaning.DbRevisionMapper;
import ru.yandex.chemodan.app.dataapi.api.deltas.cleaning.DbRevisionPojo;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.JdbcDaoUtils;
import ru.yandex.chemodan.app.dataapi.core.dao.ShardPartitionDataSource;
import ru.yandex.chemodan.app.dataapi.core.dao.ShardPartitionLocator;
import ru.yandex.chemodan.app.dataapi.core.dao.support.DataApiShardPartitionDaoSupport;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.commune.test.random.RunWithRandomTest;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.db.q.ConditionUtils;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author tolmalev
 */
public class DatabasesJdbcDaoImpl extends DataApiShardPartitionDaoSupport implements DatabasesJdbcDao {
    private final DeletedDatabasesJdbcDao deletedDatabasesJdbcDao;
    private final DatabasesJdbcDaoProperties databasesJdbcDaoProperties;

    public DatabasesJdbcDaoImpl(ShardPartitionDataSource dataSource,
                                DeletedDatabasesJdbcDao deletedDatabasesJdbcDao,
                                DatabasesJdbcDaoProperties databasesJdbcDaoProperties) {
        super(dataSource);
        this.deletedDatabasesJdbcDao = deletedDatabasesJdbcDao;
        this.databasesJdbcDaoProperties = databasesJdbcDaoProperties;
    }

    @Override
    @RunWithRandomTest
    public void insert(Database database) {
        insertBatch(database.uid, Cf.list(database));
    }

    @Override
    public void insertBatch(DataApiUserId uid, ListF<Database> databases) {
        waitToDeleteAllDatabases(uid, databases);
        JdbcDaoUtils.updateRowOrBatch(getJdbcTemplate(uid), ""
                + "INSERT INTO databases_%"
                + " (user_id, app, dbId, handle, rev, creation_time, modification_time, description, records_count, size)"
                + " VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                databases, (database) -> Cf.list(
                        database.uid.toString(),
                        database.dbAppId(),
                        database.databaseId(),
                        database.handleValue(),
                        database.rev,
                        database.meta.creationTime,
                        database.meta.modificationTime,
                        database.meta.description.getOrNull(),
                        database.meta.recordsCount, database.meta.size.toBytes()));
    }

    // https://st.yandex-team.ru/CHEMODAN-81688
    private void waitToDeleteAllDatabases(DataApiUserId uid, ListF<Database> databases) {
        if (!databasesJdbcDaoProperties.checkDeletedDatabasesEnabled()) {
            return;
        }

        Integer deletedDatabaseCount = new RetryManager<Integer>()
                .withRetryPolicy(databasesJdbcDaoProperties.getCheckDeletedDatabasesRetryPolice())
                .get(() -> deletedDatabasesJdbcDao.findDatabasesCount(uid, databases));

        if (deletedDatabaseCount != 0) {
            throw new NotDeletedDateInDatabasesException();
        }
    }

    @Override
    public void save(Database database) {
        save(database, database.rev - 1);
    }

    @Override
    public void save(Database database, long currentRevision) {
        try {
            {
                getJdbcTemplate(database.uid).updateRow(""
                        + "UPDATE databases_% SET "
                        + "rev = ?, modification_time = ?, description = ?, records_count = ?, size = ? "
                        + "WHERE user_id=? AND app=? AND dbId=? AND handle=? AND rev=?",
                        database.rev,
                        database.meta.modificationTime,
                        database.meta.description.getOrNull(),
                        database.meta.recordsCount,
                        database.meta.size.toBytes(),
                        database.uid.toString(),
                        database.dbAppId(),
                        database.databaseId(),
                        database.handleValue(),
                        currentRevision);
            }
        } catch (EmptyResultDataAccessException e) {
            throw new DatabaseRevisionMismatchException(StringUtils.format(
                    "Database {} has not {} revision or does not exist", database.databaseId(), currentRevision));
        }
    }

    @Override
    @RunWithRandomTest
    public int findDatabasesCount(DataApiUserId uid, DatabaseContext dbContext) {
        return getReadJdbcTemplate(uid).queryForInt(
                "SELECT count(*) from databases_% WHERE user_id = ? AND app = ?",
                uid.toString(), dbContext.dbAppId());
    }

    @Override
    @RunWithRandomTest
    public int findDatabasesCount(DataApiUserId uid) {
        return getReadJdbcTemplate(uid)
                .queryForInt("SELECT COUNT(*) FROM databases_% WHERE user_id = ?", uid.toString());
    }

    @Override
    @RunWithRandomTest
    public ListF<Database> find(DataApiUserId uid, boolean lockForUpdate) {
        return getReadJdbcTemplate(uid).query(
                "SELECT * from databases_% WHERE user_id = ?" + (lockForUpdate ? " FOR UPDATE" : ""),
                DatabaseMapper.FROM_REAL,
                uid.toString());
    }

    @Override
    @RunWithRandomTest
    public ListF<Database> find(DataApiUserId uid, DatabaseContext dbContext) {
        return getReadJdbcTemplate(uid).query("SELECT * from databases_% WHERE user_id = ? AND app = ?",
                DatabaseMapper.FROM_REAL,
                uid.toString(), dbContext.dbAppId());
    }

    @Override
    @RunWithRandomTest
    public Option<Database> findByHandle(DataApiUserId uid, DatabaseHandle dbHandle, DatabaseLockMode lock) {
        return lock.translateExceptions(dbHandle.dbRef, () -> getReadJdbcTemplate(uid).queryForOption(
                "SELECT * from databases_% WHERE user_id = ? AND app = ? AND handle = ?" + lock.sql(),
                DatabaseMapper.FROM_REAL,
                uid.toString(),
                dbHandle.dbAppId(),
                dbHandle.handle));
    }

    @Override
    @RunWithRandomTest
    public Option<Database> find(DataApiUserId uid, DatabaseRef dbRef) {
        return find(uid, dbRef, DatabaseLockMode.NO_LOCK);
    }

    @Override
    @RunWithRandomTest
    public Option<Database> find(DataApiUserId uid, DatabaseRef dbRef, DatabaseLockMode lock) {
        ListF<Database> dbs = find(uid, dbRef.dbContext(), Cf.list(dbRef.databaseId()), lock);

        if (dbs.size() > 1) {
            throw new IncorrectResultSizeDataAccessException(1, dbs.size());
        }
        return dbs.singleO();
    }

    @Override
    @RunWithRandomTest
    public ListF<Database> find(
            DataApiUserId uid, DatabaseContext dbContext, ListF<String> databaseIds, DatabaseLockMode lock)
    {
        SqlCondition condition = SqlCondition.trueCondition()
                .and(ConditionUtils.column("app").eq(dbContext.dbAppId()))
                .and(ConditionUtils.column("user_id").eq(uid.toString()))
                .and(ConditionUtils.column("dbId").inSet(databaseIds));

        return lock.translateExceptions(dbContext, databaseIds, () -> getReadJdbcTemplate(uid)
                .query("SELECT * FROM databases_%" + condition.whereSql() + lock.sql(),
                        DatabaseMapper.FROM_REAL,
                        condition.args()));
    }

    @Override
    public ListF<DataApiUserId> findAppUsersOrdered(
            ShardPartitionLocator shardPartition,
            DatabaseContext dbContext, Option<String> databaseId, Option<DataApiUserId> prevUid, int limit)
    {
        SqlCondition condition = ConditionUtils.column("app").eq(dbContext.dbAppId());

        if (databaseId.isPresent()) {
            condition = condition.and(ConditionUtils.column("dbId").eq(databaseId.get()));
        }
        if (prevUid.isPresent()) {
            condition = condition.and(ConditionUtils.column("user_id").gt(prevUid.get().toString()));
        }
        return getJdbcTemplate(shardPartition).query(
                "SELECT DISTINCT user_id FROM databases_%" + condition.whereSql() + " ORDER BY user_id LIMIT " + limit,
                (rs, num) -> DataApiUserId.parse(rs.getString(1)),
                condition.args());
    }

    @Override
    public ListF<Database> find(
            ShardPartitionLocator shardPartition, DatabaseRef dbRef, DatabaseCondition dbCond)
    {
        SqlCondition condition = ConditionUtils.column("app").eq(dbRef.dbAppId());
        condition = condition.and(ConditionUtils.column("dbId").eq(dbRef.databaseId()));
        condition = condition.and(dbCond.getCondition());

        return getJdbcTemplate(shardPartition).query(
                "SELECT * FROM databases_%" + condition.whereSql(), DatabaseMapper.FROM_REAL, condition.args());
    }

    @Override
    public ListF<DbRevisionPojo> findAllDatabases(
            ShardPartitionLocator shardPartition, int pageSize, Option<String> offsetHandle)
    {
        SqlCondition condition = offsetHandle.map(handle -> ConditionUtils.column("handle").gt(handle))
                .getOrElse(SqlCondition.trueCondition());
        SqlLimits limits = SqlLimits.first(pageSize);
        SqlOrder order = SqlOrder.orderByColumn("handle");
        String innerQuery = "(SELECT handle, min(rev) FROM deltas_% " + condition.whereSql() + " GROUP BY handle "
                + order.toSql() + " "
                + limits.toMysqlLimits() + ")";

        String query = "WITH deltas(handle, minRev) as " + innerQuery
                + " SELECT deltas.*, dbs.*"
                + " FROM deltas INNER JOIN databases_% dbs " + "ON deltas.handle=dbs.handle";

        return getJdbcTemplate(shardPartition).query(query, DbRevisionMapper.M, condition.args());
    }

    @Override
    public ListF<DataApiUserId> findDatabaseUsers(
            ShardPartitionLocator shardPartition, DatabaseRef dbRef, DatabaseCondition dbCond)
    {
        SqlCondition condition = ConditionUtils.column("app").eq(dbRef.dbAppId());
        condition = condition.and(ConditionUtils.column("dbId").eq(dbRef.databaseId()));
        condition = condition.and(dbCond.getCondition());

        return getJdbcTemplate(shardPartition).query(
                "SELECT DISTINCT user_id FROM databases_%" + condition.whereSql(),
                (rs, num) -> DataApiUserId.parse(rs.getString(1)),
                condition.args());
    }

    @Override
    public DatabaseHandles findHandles(DataApiUserId uid, DatabasesFilter filter) {
        SqlCondition condition = ConditionUtils.column("user_id").eq(uid.toString())
                .and(filter.toSqlCondition());
        return DatabaseHandles.fromDatabaseIdHandleTuples(filter,
                getReadJdbcTemplate(uid).query(
                        "SELECT dbId, handle FROM databases_%" + condition.whereSql(),
                        (rs, num) -> Tuple2.tuple(rs.getString(1), rs.getString(2)),
                        condition.args()
                )
        );
    }

    @Override
    public void delete(Database database) {
        delete(database.uid, database.dbContext(), Cf.list(database.databaseId()));
    }

    @Override
    public void delete(DataApiUserId uid, DatabaseContext dbContext, ListF<String> databaseIds) {
        SqlCondition condition = SqlCondition.trueCondition()
                .and(ConditionUtils.column("app").eq(dbContext.dbAppId()))
                .and(ConditionUtils.column("user_id").eq(uid.toString()))
                .and(ConditionUtils.column("dbId").inSet(databaseIds));

        int affected = getJdbcTemplate(uid).update("DELETE FROM databases_%" + condition.whereSql(), condition.args());

        if (databaseIds.size() != affected) {
            throw new IncorrectResultSizeDataAccessException(databaseIds.size(), affected);
        }
    }

    @Override
    public void updateSize(DataApiUserId uid, String handle, long rev, DataSize size) {
        getJdbcTemplate(uid).updateRow("UPDATE databases_% SET size = ? WHERE "
                + "handle = ? AND rev = ?",
                size.toBytes(), handle, rev);
    }

    @Override
    public ListF<ShardPartitionLocator> getShardPartitions() {
        return getShardPartitions("databases");
    }

    @Override
    public int findDatabasesCount() {
        return getShardPartitions()
                .map(s -> getJdbcTemplate(s).queryForInt("SELECT count(*) from databases_%")).sum(Cf.Integer);
    }

    @Override
    public ListF<ShardPartitionLocator> getShardPartitions(int shardId) {
        return super.getShardPartitions(shardId, "databases");
    }
}
