package ru.yandex.chemodan.app.dataapi.core.dao.data;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseMeta;
import ru.yandex.chemodan.app.dataapi.api.db.ref.GlobalDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserType;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
@ActivateDataApiEmbeddedPg
public class DatabasesJdbcDaoTest extends DataApiTestSupport {

    @Autowired
    private DatabasesJdbcDao dao;

    @Autowired
    private DeletedDatabasesJdbcDao deletedDatabasesJdbcDao;

    @Autowired
    private DatabasesJdbcDaoProperties databasesJdbcDaoProperties;

    @Before
    public void before() {
        databasesJdbcDaoProperties.setCheckDeletedDatabasesEnables(() -> true);
        databasesJdbcDaoProperties.setDeletedDatabasesRetryPolice(
                DatabasesJdbcDaoProperties.createRetryPolice(2, 1, 2, 2));
    }

    @Test
    public void globalDatabase() {
        DataApiUserId user = createRandomCleanUserInDefaultShard();

        // MySql stores seconds
        Instant time = new Instant(Instant.now().getMillis() / 1000 * 1000);
        testGlobalDatabase(user, time);
    }

    @Test(expected = NotDeletedDateInDatabasesException.class)
    public void testCreatingDatabaseDuringDeletion() {
        DataApiUserId user = createRandomCleanUserInDefaultShard();

        Instant time = new Instant(Instant.now().getMillis());
        Database database = createDatabase(user, time);

        dao.insert(database);
        deletedDatabasesJdbcDao.saveAsDeleted(database, time.plus(100));
        dao.insert(database);
    }

    @Test
    public void globalDatabaseYandexuid() {
        DataApiUserId user = userInitializer.createRandomCleanUserInDefaultShard(DataApiUserType.YANDEX_UID);
        testGlobalDatabase(user, Instant.now());
    }

    private void testGlobalDatabase(DataApiUserId user, Instant time) {
        Database database = createDatabase(user, time);

        dao.insert(database);
        Assert.some(database, dao.find(user, database.dbRef()));

        dao.save(database.withIncRev());
    }

    private Database createDatabase(DataApiUserId user, Instant time) {
        return new Database(
                user,
                new GlobalDatabaseRef("asdasdasdasdsad")
                        .consHandle(Random2.R.nextAlnum(30)),
                0,
                new DatabaseMeta(time, time, DataSize.fromBytes(1024), 0)
        );
    }
}
