package ru.yandex.chemodan.app.dataapi.core.dao.data;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.JdbcDaoUtils;
import ru.yandex.chemodan.app.dataapi.core.dao.ShardPartitionDataSource;
import ru.yandex.chemodan.app.dataapi.core.dao.ShardPartitionLocator;
import ru.yandex.chemodan.app.dataapi.core.dao.support.DataApiShardPartitionDaoSupport;
import ru.yandex.devtools.test.annotations.YaIgnore;
import ru.yandex.misc.db.q.SqlLimits;

/**
 * @author tolmalev
 */
@YaIgnore
public class DeletedDatabasesJdbcDaoImpl extends DataApiShardPartitionDaoSupport implements DeletedDatabasesJdbcDao {

    public DeletedDatabasesJdbcDaoImpl(ShardPartitionDataSource dataSource) {
        super(dataSource);
    }

    @Override
    public void saveAsDeleted(Database database, Instant deleteTime) {
        saveAsDeletedBatch(database.uid, Cf.list(database.withModificationTime(deleteTime)));
    }

    @Override
    public void saveAsDeletedBatch(DataApiUserId uid, ListF<Database> databases) {
        JdbcDaoUtils.updateRowOrBatch(getJdbcTemplate(uid), ""
                + "INSERT INTO deleted_databases_%"
                + " (user_id, app, dbId, handle, rev, creation_time, delete_time, description, records_count, size)"
                + " VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)",
                databases, (database) -> Cf.list(
                        database.uid.toString(),
                        database.dbAppId(),
                        database.databaseId(),
                        database.handleValue(),
                        database.rev,
                        database.meta.creationTime,
                        database.meta.modificationTime,
                        database.meta.description.getOrNull(),
                        database.meta.recordsCount,
                        database.meta.size.toBytes()));
    }

    @Override
    public void removeFromDeleted(Database database) {
        removeFromDeleted(database.uid, database.dbHandle);
    }

    @Override
    public int findDatabasesCount(DataApiUserId uid) {
        return getReadJdbcTemplate(uid)
                .queryForInt("SELECT COUNT(*) FROM deleted_databases_% WHERE user_id = ?", uid.toString());
    }

    @Override
    public int findDatabasesCount(DataApiUserId uid, ListF<Database> databases) {
        return getReadJdbcTemplate(uid).queryForInt(
                "SELECT COUNT(*) FROM deleted_databases_% WHERE user_id = ? AND handle in (?)",
                uid.toString(),
                databases.map(Database::handleValue)
        );
    }

    @Override
    public void removeFromDeleted(DataApiUserId uid, DatabaseHandle dbHandle) {
        getJdbcTemplate(uid).updateRow(""
                + "DELETE FROM deleted_databases_% "
                + "WHERE user_id = ? AND app = ? AND dbId = ? AND handle = ?",
                uid.toString(), dbHandle.dbAppId(), dbHandle.databaseId(), dbHandle.handle);
    }

    @Override
    public ListF<Database> find(DataApiUserId uid, boolean lockForUpdate) {
        return getReadJdbcTemplate(uid).query(
                "SELECT * from deleted_databases_% WHERE user_id = ?" + (lockForUpdate ? " FOR UPDATE" : ""),
                DatabaseMapper.FROM_DELETED,
                uid.toString());
    }

    @Override
    public Option<Database> find(DataApiUserId uid, DatabaseHandle dbHandle) {
        return getReadJdbcTemplate(uid).queryForOption(
                "SELECT * from deleted_databases_% WHERE user_id = ? AND app = ? AND dbId = ? AND handle = ?",
                DatabaseMapper.FROM_DELETED,
                uid.toString(),
                dbHandle.dbAppId(),
                dbHandle.databaseId(),
                dbHandle.handle);
    }

    @Override
    public ListF<Database> findDeletedBefore(ShardPartitionLocator shardPartition, Instant before, SqlLimits limits) {
        return getJdbcTemplate(shardPartition).query(
                "SELECT * FROM deleted_databases_% WHERE delete_time < ? " + limits.toMysqlLimits(),
                DatabaseMapper.FROM_DELETED, before);
    }

    @Override
    public ListF<ShardPartitionLocator> getShardPartitions() {
        return getShardPartitions("deleted_databases");
    }
}
