package ru.yandex.chemodan.app.dataapi.core.dao.support;

import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.chemodan.app.dataapi.api.db.ref.GlobalDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.dataapi.web.ReadonlyException;
import ru.yandex.misc.test.Assert;

/**
 * @author dbrylev
 */
@ActivateDataApiEmbeddedPg
public class ShardedTransactionManagerTest extends DataApiTestSupport {

    @Autowired
    private ShardedTransactionManager transactionManager;
    @Autowired
    private UserMetaManager userMetaManager;
    @Autowired
    private DataApiManager dataApiManager;

    @Test(expected = UserShardChangedDuringTransactionException.class)
    public void checkUserShardChangeBeforeCommit() {
        DataApiUserId uid = createRandomCleanUserInDefaultShard();
        int shardId = userMetaManager.findMetaUser(uid).get().getShardId();

        TransactionStatus transaction = transactionManager.getTransaction(uid);
        try {
            userMetaManager.updateShardIdAndReadOnly(uid, shardId + 1, false);
            dataApiManager.createDatabase(new UserDatabaseSpec(uid, new GlobalDatabaseRef("db")));

            transactionManager.commit(uid, transaction);

        } finally {
            transactionManager.rollback(uid, transaction);
        }
    }

    @Test(expected = ReadonlyException.class)
    public void checkReadonlyBeforeCommit() {
        DataApiUserId uid = createRandomCleanUserInDefaultShard();

        int shardId = userMetaManager.findMetaUser(uid).get().getShardId();

        TransactionStatus transaction = transactionManager.getTransaction(uid);
        try {
            dataApiManager.createDatabase(new UserDatabaseSpec(uid, new GlobalDatabaseRef("db")));
            userMetaManager.updateShardIdAndReadOnly(uid, shardId, true);

            transactionManager.commit(uid, transaction);

        } catch (Throwable e) {
            transactionManager.rollback(uid, transaction);
            throw e;
        }
    }

    @Test
    public void skipCheckReadonlyBeforeReadonlyCommit() {
        DataApiUserId uid = createRandomCleanUserInDefaultShard();
        int shardId = userMetaManager.findMetaUser(uid).get().getShardId();

        DefaultTransactionDefinition definition = new DefaultTransactionDefinition();
        definition.setReadOnly(true);

        TransactionStatus transaction = transactionManager.getTransaction(uid, definition);
        try {
            userMetaManager.updateShardIdAndReadOnly(uid, shardId, true);
            transactionManager.commit(uid, transaction);

        } catch (Throwable e) {
            transactionManager.rollback(uid, transaction);
            Assert.fail("No throwable expected but got: " + e);
        }
    }
}
