package ru.yandex.chemodan.app.dataapi.core.dao.usermeta;

import org.joda.time.DateTimeConstants;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.util.ping.PingerChecker;
import ru.yandex.chemodan.util.sharpei.SharpeiClientWithRetries;
import ru.yandex.chemodan.util.sharpei.SharpeiManager;
import ru.yandex.chemodan.util.sharpei.SharpeiShardInfo;
import ru.yandex.misc.worker.AlarmThread;

/**
 * @author tolmalev
 */
public class SharpeiUserMetaManager implements UserMetaManager, PingerChecker {

    private final SharpeiManager sharpeiManager;

    private int defaultShardId = -1;

    private final AlarmThread defaultshardUpdater;

    public SharpeiUserMetaManager(SharpeiClientWithRetries sharpeiClient, SharpeiManager sharpeiManager) {
        this.sharpeiManager = sharpeiManager;

        //remove after implementation of non-integer uids
        defaultshardUpdater = new AlarmThread(
                "SharpeiUserMetaManager-default-shard-updater", DateTimeConstants.MILLIS_PER_MINUTE, false)
        {
            protected void alarm() {
                updateDefaultShard(sharpeiClient.getShards().map(SharpeiShardInfo::getId).sorted().first());
            }
        };
        defaultshardUpdater.startGracefully();
    }

    private void updateDefaultShard(int shardId) {
        defaultShardId = shardId;
    }

    @Override
    public Option<MetaUser> findMetaUser(DataApiUserId user, boolean forRead) {
        return sharpeiManager.find(user, forRead).map(x -> MetaUser.from(user, x));
    }

    @Override
    public Option<MetaUser> findMetaUser(DataApiUserId user) {
        return findMetaUser(user, false);
    }

    @Override
    public ListF<MetaUser> findMetaUsers(ListF<DataApiUserId> userIds) {
        return userIds.flatMap(this::findMetaUser);
    }

    @Override
    public MetaUser registerIfNotExists(DataApiUserId user) {
        return MetaUser.from(user, sharpeiManager.registerIfNotExists(user));
    }

    @Override
    public void registerBatch(ListF<DataApiUserId> usersIds) {
        sharpeiManager.registerBatch(usersIds.cast());
    }

    @Override
    public void updateReadOnly(DataApiUserId user, boolean readOnly) {
        sharpeiManager.updateReadOnly(user, readOnly);
    }

    @Override
    public void updateMigrated(DataApiUserId user, DatabaseRef ref, boolean migrated) {
        sharpeiManager.updateMigrated(user, ref.toString(), migrated);
    }

    @Override
    public void updateShardIdAndReadOnly(DataApiUserId user, int toShardId, boolean readOnly) {
        sharpeiManager.updateShardIdAndReadOnly(user, toShardId, readOnly);
    }

    @Override
    public boolean isActive() {
        return defaultShardId >= 0;
    }
}
