package ru.yandex.chemodan.app.dataapi.core.datasources;

import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.datasource.AuxiliaryDataSource;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceRegistry;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceType;
import ru.yandex.chemodan.app.dataapi.api.datasource.SpecificDataSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class BasicDataSourceRegistry implements DataSourceRegistry {
    private final DataSourceTypeRegistry typeRegistry;

    private final MapF<DataSourceType, SpecificDataSource> dsToType;

    private final MapF<DatabaseRef, DataSourceType> refToType;

    private final DataSourceType defaultType;

    public BasicDataSourceRegistry(DataSourceTypeRegistry typeRegistry, DiskDataSource diskDataSource,
            ListF<SpecificDataSource> dataSources)
    {
        this(typeRegistry, dataSources.plus1(diskDataSource).unique(), DataSourceType.DISK);
    }

    public BasicDataSourceRegistry(DataSourceTypeRegistry typeRegistry, CollectionF<SpecificDataSource> dataSources,
            DataSourceType defaultType)
    {
        this.typeRegistry = typeRegistry;
        this.dsToType = dataSources.toMapMappingToKey(SpecificDataSource::type);
        this.refToType = dataSources.filterMap(BasicDataSourceRegistry::getAuxDataSourceO)
                .zipWithFlatMap(AuxiliaryDataSource::getDatabaseRefs)
                .map1(SpecificDataSource::type)
                .invert()
                .toMap();
        this.defaultType = defaultType;
    }

    private static Option<AuxiliaryDataSource> getAuxDataSourceO(SpecificDataSource ds) {
        return ds instanceof AuxiliaryDataSource ? Option.of((AuxiliaryDataSource) ds) : Option.empty();
    }

    @Override
    public SpecificDataSource getDataSource(UserDatabaseSpec databaseSpec) {
        return getDataSource(getType(databaseSpec.databaseRef()));
    }

    public SpecificDataSource getDataSource(DatabaseRef databaseRef) {
        return getDataSource(getType(databaseRef));
    }

    public SpecificDataSource getDataSource(DataSourceType type) {
        return dsToType.getTs(type);
    }

    private DataSourceType getType(DatabaseRef databaseRef) {
        return typeRegistry.getTypeO(databaseRef)
                    .orElse(() -> refToType.getO(databaseRef))
                    .getOrElse(defaultType);
    }
}
