package ru.yandex.chemodan.app.dataapi.core.datasources;

import org.junit.Assert;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.function.Function;
import ru.yandex.chemodan.app.dataapi.api.datasource.AuxiliaryDataSource;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceSession;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceType;
import ru.yandex.chemodan.app.dataapi.api.datasource.SpecificDataSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;

import static org.mockito.Matchers.any;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class BasicDataSourceRegistryTest {
    private static final DataApiUserId TEST_USER = new DataApiPassportUserId(1);

    private static final AppDatabaseRef PASSPORT_DB_REF = new AppDatabaseRef("pass", "pass");

    private static final SpecDataSource diskDataSource = new SpecDataSource(DataSourceType.DISK);

    private static final AuxDataSource passportDataSource =
            new AuxDataSource(DataSourceType.PASSPORT, PASSPORT_DB_REF);

    private static final DataSourceTypeRegistry typeRegistry =
            mockWithOptionReturnType(DataSourceTypeRegistry.class, mock -> mock.getTypeO(any()));

    private static final BasicDataSourceRegistry dataSourceRegistry = new BasicDataSourceRegistry(
            typeRegistry,
            Cf.list(diskDataSource, passportDataSource),
            DataSourceType.DISK
    );

    @Test
    public void testDefault() {
        Assert.assertSame(
                diskDataSource,
                dataSourceRegistry.getDataSource(
                        new UserDatabaseSpec(TEST_USER, new AppDatabaseRef("other", "other"))
                )
        );
    }

    @Test
    public void testAux() {
        Assert.assertSame(
                passportDataSource,
                dataSourceRegistry.getDataSource(new UserDatabaseSpec(TEST_USER, PASSPORT_DB_REF))
        );
    }

    private static class SpecDataSource implements SpecificDataSource {
        final DataSourceType type;

        SpecDataSource(DataSourceType type) {
            this.type = type;
        }

        @Override
        public DataSourceType type() {
            return type;
        }

        @Override
        public DataSourceSession openSession(UserDatabaseSpec databaseSpec) {
            return null;
        }
    }

    private static class AuxDataSource extends SpecDataSource implements AuxiliaryDataSource {
        final ListF<DatabaseRef> databaseRefs;

        AuxDataSource(DataSourceType type, DatabaseRef... databaseRefs) {
            super(type);
            this.databaseRefs = Cf.list(databaseRefs);
        }

        @Override
        public ListF<DatabaseRef> getDatabaseRefs() {
            return databaseRefs;
        }
    }

    private static <M> M mockWithOptionReturnType(Class<M> mockClass, Function<M, Option<?>> whenMethod) {
        M mock = Mockito.mock(mockClass);
        Mockito.when(whenMethod.apply(mock))
                .thenReturn(Option.empty());
        return mock;
    }
}
