package ru.yandex.chemodan.app.dataapi.core.datasources;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceType;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefMarshallerUnmarshaller;
import ru.yandex.commune.zk2.ZkPath;
import ru.yandex.commune.zk2.client.ZkManager;
import ru.yandex.commune.zk2.primitives.registry.ZkRegistry;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.BenderParserSerializer;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.BenderSettings;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class DataSourceTypeRegistry {
    private static final BenderParserSerializer<RefTypePojo> parserSerializer = Bender.cons(RefTypePojo.class,
            new BenderConfiguration(
                    new BenderSettings(),
                    CustomMarshallerUnmarshallerFactoryBuilder.cons()
                            .add(DatabaseRef.class,
                                    new DatabaseRefMarshallerUnmarshaller(),
                                    new DatabaseRefMarshallerUnmarshaller())
                            .build()
            )
    );

    private final ZkRegistry<DatabaseRef, RefTypePojo> zkRegistry;

    public DataSourceTypeRegistry(ZkPath path) {
        this.zkRegistry = new ZkRegistry<>(path,
                parserSerializer,
                RefTypePojo::databaseRef,
                DatabaseRef::toString);
    }

    public void addTo(ZkManager zkManager) {
        zkManager.addClient(zkRegistry);
    }

    public Option<DataSourceType> getTypeO(DatabaseRef databaseRef) {
        return getO(databaseRef)
                .filterNot(RefTypePojo::isMigrating)
                .map(RefTypePojo::currentType);
    }

    public Option<DataSourceType> getMigrationTypeO(DatabaseRef databaseRef) {
        return getO(databaseRef)
                .filterMap(RefTypePojo::migrationType);
    }

    public void setMigrating(DatabaseRef databaseRef, DataSourceType currentType, DataSourceType migrationType) {
        Option<DataSourceType> storedCurrentTypeO = getTypeO(databaseRef);
        if (storedCurrentTypeO.isPresent() && storedCurrentTypeO.get() != currentType) {
            throw new IllegalArgumentException("Specified current type = " + currentType +
                    " doesn't equal stored = " + storedCurrentTypeO.get());
        }

        zkRegistry.put(RefTypePojo.cons(databaseRef, currentType, migrationType));
    }

    public void setMigrated(DatabaseRef databaseRef) {
        zkRegistry.put(zkRegistry.get(databaseRef).toMigrated());
    }

    public Option<RefTypePojo> getO(DatabaseRef databaseRef) {
        return zkRegistry.getO(databaseRef);
    }

    @BenderBindAllFields
    public static class RefTypePojo {
        final DatabaseRef databaseRef;

        final DataSourceType currentType;

        final Option<DataSourceType> migrationType;

        RefTypePojo(DatabaseRef databaseRef, DataSourceType currentType, Option<DataSourceType> migrationType) {
            this.databaseRef = databaseRef;
            this.currentType = currentType;
            this.migrationType = migrationType;
        }

        static RefTypePojo cons(DatabaseRef databaseRef, DataSourceType currentType, DataSourceType migrationType) {
            return new RefTypePojo(databaseRef, currentType, Option.of(migrationType));
        }

        DatabaseRef databaseRef() {
            return databaseRef;
        }

        public DataSourceType currentType() {
            return currentType;
        }

        public Option<DataSourceType> migrationType() {
            return migrationType;
        }

        public boolean isMigrating() {
            return migrationType.isPresent();
        }

        RefTypePojo toMigrated() {
            return new RefTypePojo(databaseRef, migrationType.get(), Option.empty());
        }
    }
}
