package ru.yandex.chemodan.app.dataapi.core.datasources.disk;

import java.util.Comparator;
import java.util.LinkedHashSet;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Stream;

import net.jodah.failsafe.RetryPolicy;
import org.joda.time.Instant;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.IncorrectResultSizeDataAccessException;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.impl.AbstractPrefetchingIterator;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContext;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DatabaseCondition;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.DatabaseRecord;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceSession;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceType;
import ru.yandex.chemodan.app.dataapi.api.datasource.DsSessionTxManager;
import ru.yandex.chemodan.app.dataapi.api.datasource.ExtendedDataSource;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseExistsException;
import ru.yandex.chemodan.app.dataapi.api.db.filter.DatabasesFilter;
import ru.yandex.chemodan.app.dataapi.api.db.filter.DatabasesFilterOrAll;
import ru.yandex.chemodan.app.dataapi.api.db.filter.DatabasesFilterSource;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandles;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseAliasType;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.internalpublic.PublicDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.revision.DatabaseRefRevisions;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltasJdbcDao;
import ru.yandex.chemodan.app.dataapi.api.deltas.ModifiedRecordsPojo;
import ru.yandex.chemodan.app.dataapi.api.deltas.OutdatedChangeException;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.api.user.UsersIterationKey;
import ru.yandex.chemodan.app.dataapi.api.user.UsersPojo;
import ru.yandex.chemodan.app.dataapi.apps.CompositeApplicationManager;
import ru.yandex.chemodan.app.dataapi.core.DataApiStats;
import ru.yandex.chemodan.app.dataapi.core.dao.ForcedUserShardInfoHolder;
import ru.yandex.chemodan.app.dataapi.core.dao.ShardPartitionLocator;
import ru.yandex.chemodan.app.dataapi.core.dao.UserShardInfo;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DataRecordsJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabaseLockMode;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DeletedDatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.support.ShardedTransactionManager;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.chemodan.app.dataapi.web.UserNotFoundException;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.chemodan.util.retry.RetryManager;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
public class DiskDataSource implements ExtendedDataSource {

    private static final Logger logger = LoggerFactory.getLogger(DiskDataSource.class);
    private static final int QUERY_IN_CLAUSE_SIZE = 1000;

    private final DatabasesJdbcDao databasesDao;
    private final DataRecordsJdbcDao dataRecordsDao;
    private final DeletedDatabasesJdbcDao deletedDatabasesJdbcDao;
    private final DeltasJdbcDao deltasDao;
    private final ShardedTransactionManager transactionManager;
    private final CompositeApplicationManager appManager;
    private final UserMetaManager userMetaManager;

    public DiskDataSource(DatabasesJdbcDao databasesDao,
            DeletedDatabasesJdbcDao deletedDatabasesJdbcDao,
            DeltasJdbcDao deltasDao,
            ShardedTransactionManager transactionManager,
            DataRecordsJdbcDao protobufRecordsDao,
            CompositeApplicationManager appManager,
            UserMetaManager userMetaManager)
    {
        this.databasesDao = databasesDao;
        this.deletedDatabasesJdbcDao = deletedDatabasesJdbcDao;
        this.deltasDao = deltasDao;
        this.transactionManager = transactionManager;
        this.dataRecordsDao = protobufRecordsDao;
        this.appManager = appManager;
        this.userMetaManager = userMetaManager;
    }

    @Override
    public DataSourceType type() {
        return DataSourceType.DISK;
    }

    // database methods

    public ListF<Database> listDatabases(DataApiUserId uid, DatabasesFilterOrAll filter) {
        switch (filter.getType()) {
            case CONTEXT:
                return databasesDao.find(uid, filter.dbContext());

            case REFS:
                return databasesDao.find(uid, filter.dbContext(), filter.databaseIds(), DatabaseLockMode.NO_LOCK);

            case ALL:
                return databasesDao.find(uid, false);

            default:
                throw new IllegalArgumentException("Unknown database filter specified: " + filter);
        }
    }

    @Override
    public ListF<Database> listDeletedDatabases(DataApiUserId uid) {
        return deletedDatabasesJdbcDao.find(uid, false);
    }

    public void deleteDatabases(DataApiUserId uid, DatabasesFilter filter, DatabaseDeletionMode mode,
                                Option<ChunkRateLimiter> rateLimiter) {
        TransactionStatus transaction;
        try {
            transaction = transactionManager.getTransaction(uid);
        } catch (UserNotFoundException e) {
            return;
        }

        try {
            ListF<Database> databases = listDatabases(uid, filter.toAllFilter());

            if (mode == DatabaseDeletionMode.MARK_DELETED) {
                moveDatabasesToDeleted(databases, Instant.now());
            } else {
                databasesDao.delete(uid, filter.dbContext(), databases.map(DatabaseRefSource::databaseId));
                if (rateLimiter.isPresent()) {
                    cleanupDatabasesData(uid, databases.map(db -> db.dbHandle), rateLimiter.get());
                } else {
                    cleanupDatabasesData(uid, databases.map(db -> db.dbHandle));
                }
            }
            transactionManager.commit(uid, transaction);
        } catch (Throwable e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            transactionManager.rollback(uid, transaction);
            throw e;
        }
    }

    void moveDatabasesToDeleted(ListF<Database> databases, Instant deletionTime) {
        if (databases.isEmpty()) {
            return;
        }

        Validate.hasSize(1, databases.stableUniqueBy(db -> Tuple2.tuple(db.appNameO(), db.uid)));

        DatabaseContext dbContext = databases.first().dbContext();
        DataApiUserId uid = databases.first().uid;

        ListF<String> databaseIds = databases.map(Database::databaseId);
        try {
            databasesDao.delete(uid, dbContext, databaseIds);
        } catch (IncorrectResultSizeDataAccessException e) {
            // In case, when database not exists
            throw new NotFoundException("Some of databases not found " + dbContext.appNameO() + "." + databaseIds);
        }
        deletedDatabasesJdbcDao.saveAsDeletedBatch(uid, databases.map(db -> db.withModificationTime(deletionTime)));
        // records will be removed sometime later in completelyRemoveDatabase()
    }

    private void cleanupDatabasesData(DataApiUserId uid, ListF<DatabaseHandle> dbHandles) {
        ListF<String> handles = dbHandles.map(handle -> handle.handle);
        dataRecordsDao.deleteAllRecordFromDatabases(uid, handles);
        deltasDao.deleteAllForDatabases(uid, handles);
    }

    private void cleanupDatabasesData(DataApiUserId uid, ListF<DatabaseHandle> dbHandles, ChunkRateLimiter rateLimiter) {
        ListF<String> handles = dbHandles.map(handle -> handle.handle);
        dataRecordsDao.deleteAllRecordFromDatabases(uid, handles, rateLimiter);
        deltasDao.deleteAllForDatabases(uid, handles, rateLimiter);
    }


    // records

    private DatabaseHandles getDatabaseHandles(DataApiUserId uid, DatabasesFilter filter) {
        return filter.dbHandlesO()
                .getOrElse(() -> doGetDatabaseHandles(uid, filter));
    }

    private DatabaseHandles doGetDatabaseHandles(DataApiUserId uid, DatabasesFilter filter) {
        return consPermanentDbHandlesO(uid, filter)
                .getOrElse(() -> databasesDao.findHandles(uid, filter))
                .excludeRefs(filter.excludeDbRefs);
    }

    private Option<DatabaseHandles> consPermanentDbHandlesO(DataApiUserId uid, DatabasesFilter filter) {
        return filter.dbRefsSrcO()
                .filterMap(refsSrc -> appManager.consPermanentDbHandlesO(uid, refsSrc));
    }

    public ListF<DataRecord> getDataRecords(DataApiUserId uid, DatabasesFilter filter) {
        return getRecords(uid, filter);
    }

    private ListF<DataRecord> getRecords(DataApiUserId uid, DatabasesFilter filter) {
        DatabaseHandles dbHandles = getDatabaseHandles(uid, filter);
        if (dbHandles.isEmpty()) {
            return Cf.list();
        }

        return filter.getRevisionsZippedWith(dbHandles)
                .map(revisions ->
                        dataRecordsDao.findByHandlesAndMinRevisionsOrderedByRev(uid, revisions, filter.getLimits()))
                .getOrElse(() ->
                        dataRecordsDao.findByDatabaseHandles(uid, dbHandles));
    }

    @Override
    public ModifiedRecordsPojo getRecordsModifiedSinceRevisions(DataApiUserId uid, DatabaseRefRevisions sinceRevisions,
            Option<Integer> limitO)
    {
        DatabasesFilter filter = sinceRevisions.toDbsFilter()
                .withLimits(limitO.map(limit -> limit + 1)
                        .map(SqlLimits::first)
                        .getOrElse(SqlLimits.all())
                );
        ListF<DatabaseRecord> records = getDataRecords(uid, filter)
                .map(DataRecord::toDatabaseRecord);
        Option<Integer> takeCountO = limitO.filter(limit -> records.size() > limit);
        return ModifiedRecordsPojo.consFromRecordsSinceRevisions(
                takeCountO.isPresent()
                        ? records.take(takeCountO.get())
                        : records,
                sinceRevisions,
                takeCountO.isPresent()
        );
    }


    // delta

    public void removeDelta(long rev, Database database) {
        deltasDao.delete(database.uid, database.handleValue(), rev);
    }

    public void removeDeltasBefore(long rev, Database database, ChunkRateLimiter rateLimiter) {
        deltasDao.deleteBeforeRevision(database.uid, database.handleValue(), rev, rateLimiter);
    }

    public UsersPojo getAppUsers(
            DatabaseContext dbContext, Option<String> dbId,
            Option<UsersIterationKey> iterationKey, int limit)
    {
        ListF<ShardPartitionLocator> parts = databasesDao.getShardPartitions()
                .sorted(Comparator.comparingInt(ShardPartitionLocator::getPartNo).thenComparingInt(spl -> spl.shardId));

        int keyShardId = iterationKey.map(k -> k.shardId).orElse(parts.first().shardId);
        int keyPartNo = iterationKey.map(k -> k.partitionNo).orElse(0);

        Option<DataApiUserId> prevUid = iterationKey.map(k -> k.prevUid);

        SetF<DataApiUserId> result = Cf.x(new LinkedHashSet<>());

        for (ShardPartitionLocator part: parts) {
            int partNo = part.getPartNo(), shardId = part.shardId;

            if (partNo > keyPartNo || partNo == keyPartNo && shardId >= keyShardId) {
                for (;;) {
                    int partLimit = limit - result.size() + 100; // in case of duplication

                    ListF<DataApiUserId> users = databasesDao.findAppUsersOrdered(part, dbContext, dbId, prevUid, partLimit);
                    ListF<DataApiUserId> taken = users.filterNot(result::containsTs).take(limit - result.size());

                    result.addAll(taken);

                    if (result.size() >= limit) {
                        UsersIterationKey key = new UsersIterationKey(shardId, partNo, taken.last());
                        return new UsersPojo(limit, result, Option.of(key));
                    }
                    if (users.size() < partLimit) {
                        break;
                    }
                    prevUid = users.lastO();
                }
                prevUid = Option.empty();
            }
        }
        return new UsersPojo(limit, result, Option.empty());
    }

    public Stream<DataApiUserId> getDatabaseUsersStream(DatabaseRef ref) {
        return databasesDao
                .getShardPartitions()
                .parallelStream()
                .flatMap(part -> databasesDao.find(part, ref, DatabaseCondition.all()).stream())
                .map(db -> db.uid);
    }

    public IteratorF<Database> getDatabases(
            DatabaseRefSource dbRefSrc, DatabaseCondition dbCondition)
    {
        return databasesDao.getShardPartitions().iterator()
                .flatMap(part -> databasesDao.find(part, dbRefSrc.dbRef(), dbCondition).iterator());
    }

    public DatabaseUsersByPartitionIterator getDatabaseUsers(DatabaseRef dbRef) {
        return new DatabaseUsersByPartitionIterator(dbRef);
    }

    public class DatabaseUsersByPartitionIterator extends AbstractPrefetchingIterator<ListF<DataApiUserId>> {
        private final DatabaseRef dbRef;

        private final DatabaseCondition dbCond;

        private final IteratorF<ShardPartitionLocator> partitions;

        private final RetryManager<ListF<DataApiUserId>> retryManager;

        private final boolean safe;

        private DatabaseUsersByPartitionIterator(DatabaseRef dbRef) {
            this(dbRef, DatabaseCondition.all(),
                    databasesDao.getShardPartitions()
                            .iterator(),
                    new RetryManager<>(),
                    false
            );
        }

        private DatabaseUsersByPartitionIterator(
                DatabaseRef dbRef,
                DatabaseCondition dbCond,
                IteratorF<ShardPartitionLocator> partitions,
                RetryManager<ListF<DataApiUserId>> retryManager,
                boolean safe)
        {
            this.dbRef = dbRef;
            this.dbCond = dbCond;
            this.partitions = partitions;
            this.retryManager = retryManager;
            this.safe= safe;
        }

        public DatabaseUsersByPartitionIterator withDbCond(DatabaseCondition dbCond) {
            return new DatabaseUsersByPartitionIterator(dbRef, dbCond, partitions, retryManager, safe);
        }

        public DatabaseUsersByPartitionIterator withRetryPolicy(RetryPolicy retryPolicy) {
            return new DatabaseUsersByPartitionIterator(dbRef, dbCond, partitions,
                    retryManager.withRetryPolicy(retryPolicy),
                    safe
            );
        }

        public DatabaseUsersByPartitionIterator safe() {
            return new DatabaseUsersByPartitionIterator(dbRef, dbCond, partitions, retryManager, true);
        }

        @Override
        protected Option<ListF<DataApiUserId>> fetchNext() {
            return partitions.nextO()
                    .map(this::doFetchNext);
        }

        private ListF<DataApiUserId> doFetchNext(ShardPartitionLocator shardAndPart) {
            return retryManager
                    .withLogging(String.format("Getting users from partition [%s, %s] for database %s",
                            shardAndPart.shardId, shardAndPart.partition, dbRef)
                    )
                    .get(safe, () -> fetchDbUsers(shardAndPart), Cf::list);
        }

        private ListF<DataApiUserId> fetchDbUsers(ShardPartitionLocator shardAndPart) {
            return databasesDao.findDatabaseUsers(shardAndPart, dbRef, dbCond);
        }
    }

    // deleted databases methods

    public Option<Database> getDeletedDatabaseO(DataApiUserId uid, DatabaseHandle dbHandle) {
        return deletedDatabasesJdbcDao.find(uid, dbHandle);
    }

    @Override
    public void completelyRemoveDatabasesDeletedBefore(Instant before) {
        int limit = 100;
        for (ShardPartitionLocator shardPartition : deletedDatabasesJdbcDao.getShardPartitions()) {
            ListF<Database> databases;
            int skip = 0;
            do {
                databases = deletedDatabasesJdbcDao.findDeletedBefore(
                        shardPartition, before, SqlLimits.range(skip, limit));

                for (Database db : databases) {
                    ForcedUserShardInfoHolder.set(new UserShardInfo(db.uid, false, shardPartition.shardId));
                    try {
                        completelyRemoveDatabase(db);
                    } catch (IncorrectResultSizeDataAccessException e) {
                        String dbName = Cf.list(db.uid, db.appNameO(), db.databaseId(), db.handleValue())
                                .mkString(".");
                        logger.error("Failed to remove database {}", dbName, e);
                        skip++;
                    } finally {
                        ForcedUserShardInfoHolder.remove();
                    }
                }
            } while (databases.size() >= limit);
        }
    }

    private void completelyRemoveDatabase(Database database) {
        completelyRemoveDatabase(database.uid, database.dbHandle);
    }

    private void completelyRemoveDatabase(DataApiUserId uid, DatabaseHandle dbHandle) {
        TransactionStatus transaction = transactionManager.getTransaction(uid);
        try {
            deletedDatabasesJdbcDao.removeFromDeleted(uid, dbHandle);
            cleanupDatabasesData(uid, Cf.list(dbHandle));

            transactionManager.commit(uid, transaction);
        } catch (Throwable e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            transactionManager.rollback(uid, transaction);
            throw e;
        }
    }

    @SuppressWarnings("unused")
    public void restoreDeletedDatabase(DataApiUserId uid, DatabaseHandle dbHandle) {
        TransactionStatus transaction = transactionManager.getTransaction(uid);
        try {
            Database database = getDeletedDatabaseO(uid, dbHandle)
                    .getOrThrow("Database not found " + dbHandle.dbRef());

            deletedDatabasesJdbcDao.removeFromDeleted(uid, dbHandle);
            databasesDao.insert(database.touch());

            transactionManager.commit(uid, transaction);
        } catch (Throwable e) {
            ExceptionUtils.throwIfUnrecoverable(e);
            transactionManager.rollback(uid, transaction);
            throw e;
        }
    }

    @SuppressWarnings("unused")
    public void recountSize(DataApiUserId uid, DatabaseRef dbRef) {
        Database database = openSession(new UserDatabaseSpec(uid, dbRef))
                .getDatabase();
        ListF<DataRecord> records = getRecords(uid, database);
        DataSize size = records.map(DataRecord::getSize)
                .foldLeft(DataSize.ZERO, DataSize::plus);
        databasesDao.updateSize(uid, database.handleValue(), database.rev, size);
    }

    @Override
    public Session openSession(UserDatabaseSpec databaseSpec) {
        return new Session(databaseSpec);
    }

    /**
     * DEFAULT METHODS
     */

    // database

    // admin-only
    @Override
    public ListF<Database> listDatabases(DataApiUserId uid) {
        return listDatabases(uid, DatabasesFilterOrAll.all());
    }

    public ListF<Database> listDatabases(DataApiUserId uid, DatabasesFilterSource filterSrc) {
        return listDatabases(uid, filterSrc.toDbsFilter().toAllFilter());
    }


    public void deleteDatabases(DataApiUserId uid, DatabasesFilterSource filterSrc) {
        deleteDatabases(uid, filterSrc, DatabaseDeletionMode.MARK_DELETED);
    }

    @Override
    public void deleteDatabases(DataApiUserId uid, DatabasesFilterSource filterSrc, DatabaseDeletionMode mode) {
        deleteDatabases(uid, filterSrc.toDbsFilter(), mode, Option.empty());
    }

    @Override
    public void deleteDatabases(DataApiUserId uid, DatabasesFilterSource filterSrc, DatabaseDeletionMode mode,
                                ChunkRateLimiter rateLimiter) {
        deleteDatabases(uid, filterSrc.toDbsFilter(), mode, Option.of(rateLimiter));
    }


    // record

    public ListF<DataRecord> getRecords(DataApiUserId uid, DatabasesFilterSource filterSource) {
        return getDataRecords(uid, filterSource.toDbsFilter());
    }

    public class Session implements DataSourceSession {
        private final UserDatabaseSpec userDbRef;

        private Option<Option<Database>> lastObtainedDatabase = Option.empty();

        private final SessionTxManager transactionManager;

        Session(UserDatabaseSpec userDbRef) {
            this.userDbRef = userDbRef;
            this.lastObtainedDatabase = userDbRef.databaseO().map(Option::of);
            this.transactionManager = new SessionTxManager(this);
        }

        @Override
        public UserDatabaseSpec databaseSpec() {
            return userDbRef;
        }

        @Override
        public Option<Database> getDatabaseO() {
            return getDatabaseWithoutLockingO();
        }

        @Override
        public void deleteDatabase(DatabaseDeletionMode deletionMode) {
            if (userDbRef.dbHandleO().isPresent()) {
                deleteDatabases(uid(), userDbRef.dbHandleO().get(), deletionMode);
            } else {
                deleteDatabases(uid(), databaseRef(), deletionMode);
            }
        }

        @Override
        public void deleteDatabase(DatabaseDeletionMode deletionMode, ChunkRateLimiter rateLimiter) {
            if (userDbRef.dbHandleO().isPresent()) {
                deleteDatabases(uid(), userDbRef.dbHandleO().get(), deletionMode, rateLimiter);
            } else {
                deleteDatabases(uid(), databaseRef(), deletionMode, rateLimiter);
            }
        }

        @Override
        public Database getOrCreateDatabase() {
            return getDatabaseO()
                .getOrElse(this::safeCreateDatabase);
        }

        private Database safeCreateDatabase() {
            return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.RW_M,
                    () -> {
                        try {
                            return createDatabase();
                        } catch (DatabaseExistsException e) {
                            // XXX: do smth better
                            lastObtainedDatabase = Option.empty();
                            return getDatabase();
                        }
                    });
        }

        @Override
        public Database createDatabase() {
            return createDatabase(Option.empty());
        }

        @Override
        public Database createDatabaseWithDescription(String title) {
            return createDatabase(Option.of(title));
        }

        private Database createDatabase(Option<String> description) {
            try {
                return createDatabaseInternal(description);
            } catch (UserNotFoundException e) {
                userMetaManager.registerIfNotExists(uid());
                return createDatabaseInternal(description);
            }
        }

        private Database createDatabaseInternal(Option<String> description) {
            try {
                Database database = Database.consNew(uid(), generateHandle(), Instant.now(), description);
                saveNewDatabase(database);
                return database;
            } catch (DataIntegrityViolationException e) {
                throw userDbRef.consExistsException();
            }
        }

        private DatabaseHandle generateHandle() {
            return databaseAlias().aliasType() == DatabaseAliasType.PUBLIC
                    ? ((PublicDatabaseAlias) databaseAlias()).consHandle()
                    : appManager.createDatabaseHandle(uid(), databaseRef());
        }

        @Override
        public Database setDatabaseDescription(Option<String> newTitle) {
            Database database = getDatabase()
                    .withDescription(newTitle);
            long currentRev = database.rev;
            saveExistingDatabase(database, currentRev);
            return database;
        }

        @Override
        public Database fixDatabaseRevision(long rev, long currentRev) {
            Database database = getDatabase()
                    .withRev(rev).withModificationTime(Instant.now());

            saveExistingDatabase(database, currentRev);
            return database;
        }

        @Override
        public void onDatabaseUpdate(long rev) {
            throw new UnsupportedOperationException("Could not set revision on native database " + userDbRef);
        }

        private void saveNewDatabase(Database database) {
            databasesDao.insert(database);
            setLastObtainedDatabase(database);
        }

        private void saveExistingDatabase(Database database, long currentRev) {
            databasesDao.save(database, currentRev);
            setLastObtainedDatabase(database);
            step("save-database");
        }

        private void setLastObtainedDatabase(Option<Database> database) {
            lastObtainedDatabase = Option.of(database);
        }

        private void setLastObtainedDatabase(Database database) {
            lastObtainedDatabase = Option.of(Option.of(database));
        }

        @Override
        public Delta getDelta(long rev) {
            return deltasDao.find(uid(), getHandle(), rev)
                        .getOrThrow(() -> NotFoundException.consDeltaNotFound(rev));
        }

        @Override
        public ListF<Delta> listDeltas(long fromRev, int limit) {
            return deltasDao.findAfterRevision(uid(), getHandle(), fromRev, limit);
        }

        @Override
        public Option<Snapshot> getSnapshotO(RecordsFilter filter) {
            Option<Database> databaseO = getDatabaseO();
            return databaseO.map(database -> {
                if (database.meta.recordsCount == 0) {
                    return new Snapshot(database, Cf.list());
                }
                // won't fail because handle exists
                return getDataRecordsO(filter)
                        .map(records -> new Snapshot(database, records))
                        .get();
            });
        }

        @Override
        public Option<ListF<DataRecord>> getDataRecordsO(RecordsFilter filter) {
            return getHandleO()
                        .map(handle -> dataRecordsDao.find(uid(),
                                handle,
                                filter.getCollectionIdCond(),
                                filter.getRecordIdCond(),
                                filter.getRecordCond(),
                                filter.getRecordOrder(),
                                filter.limits()
                        ));
        }

        @Override
        public ListF<DataRecord> getDataRecords(RecordsFilter filter) {
            // Если мы можем получить handle без запроса в базу, делаем это
            // Если нет - идем за объектами одним запросом с вложенным SELECT-ом
            if (hasHandleInMemory()) {
                return getDataRecordsO(filter).getOrElse(Cf.list());
            } else {
                return dataRecordsDao.find(uid(),
                        databaseRef(),
                        filter.getCollectionIdCond(),
                        filter.getRecordIdCond(),
                        filter.getRecordCond(),
                        filter.getRecordOrder(),
                        filter.limits());
            }
        }

        private boolean hasHandleInMemory() {
            return userDbRef.dbHandleO().isPresent()
                    || lastObtainedDatabase.isPresent()
                    || appManager.consPermanentDbHandleO(uid(), databaseRef()).isPresent();
        }

        @Override
        public int getDataRecordsCount(RecordsFilter filter) {
            if (hasHandleInMemory()) {
                return getHandleO()
                        .map(handle -> dataRecordsDao.count(uid(),
                                handle,
                                filter.getCollectionIdCond(),
                                filter.getRecordIdCond(),
                                filter.getRecordCond())
                        )
                        .getOrElse(0);
            } else {
                return dataRecordsDao
                        .count(uid(), databaseRef(), filter.getCollectionIdCond(), filter.getRecordIdCond());
            }
        }

        private Option<Database> retrieveDatabaseLockedForUpdateO() {
            DatabaseLockMode mode = userDbRef.settings().isNowaitLock()
                    ? DatabaseLockMode.FOR_UPDATE_NOWAIT
                    : DatabaseLockMode.FOR_UPDATE;

            return retrieveDatabaseO(mode);
        }

        private Option<Database> getDatabaseWithoutLockingO() {
            return lastObtainedDatabase.getOrElse(this::retrieveDatabaseWithoutLockingO);
        }

        public Option<Database> retrieveDatabaseWithoutLockingO() {
            return retrieveDatabaseO(DatabaseLockMode.NO_LOCK);
        }

        private Option<Database> retrieveDatabaseO(DatabaseLockMode lock) {
            Option<Database> database = userDbRef.dbHandleO().isPresent()
                                        ? retrieveDatabaseByHandleO(userDbRef.dbHandleO().get(), lock)
                                        : retrieveDatabaseByRefO(lock);
            setLastObtainedDatabase(database);
            return database;
        }

        private Option<Database> retrieveDatabaseByHandleO(DatabaseHandle dbHandle, DatabaseLockMode lock) {
            return databasesDao.findByHandle(uid(), dbHandle, lock);
        }

        private Option<Database> retrieveDatabaseByRefO(DatabaseLockMode lock) {
            return doRetrieveDatabaseO(lock);
        }

        private Option<Database> doRetrieveDatabaseO(DatabaseLockMode lock) {
            return databasesDao.find(uid(), databaseRef().dbRef(), lock);
        }

        @Override
        public void save(DatabaseChange change) {
            saveDeltas(change);
            saveRecordChanges(change);
            savePatchedDatabase(change);
        }

        private void saveDeltas(DatabaseChange change) {
            if (change.deltas.isEmpty()) {
                return;
            }

            Database database = change.patchedDatabase();
            for (Delta delta : change.deltas) {
                try {
                    deltasDao.insert(database.uid, database.handleValue(), delta);
                } catch (DataIntegrityViolationException e) {
                    throw new OutdatedChangeException(
                            StringUtils.format("Delta with rev={} was already applied for database={}",
                                    delta.rev,
                                    database.handleValue()));
                }
            }

            step("save-delta");
        }

        private void saveRecordChanges(DatabaseChange change) {
            dataRecordsDao.deleteRecordsBatched(uid(), change.getDeletedIds());
            dataRecordsDao.insertBatched(uid(), databaseRef(), change.getNewRecords());
            dataRecordsDao.updateBatched(uid(), change.getUpdatedRecords());
            step("apply-changes-in-dao");
        }

        private void savePatchedDatabase(DatabaseChange change) {
            saveExistingDatabase(change.patchedDatabase(), change.sourceDatabase().rev);
        }

        @Override
        public ListF<DataRecord> getDataRecordsByIds(SetF<DataRecordId> recordIds) {
            Database database = getDatabase();
            if (database.rev == 0) {
                return Cf.list();
            }

            ListF<DataRecord> result = recordIds.iterator()
                    .paginate(QUERY_IN_CLAUSE_SIZE)
                    .map(batch -> dataRecordsDao.findRecords(uid(), database.dbHandle, batch))
                    .flatMap(ListF::iterator)
                    .toList();
            step("find-records");

            return result;
        }

        public void deleteAllRecords() {
            dataRecordsDao.deleteAllRecordFromDatabases(
                    uid(),
                    Cf.list(getHandle().handleValue())
            );
        }

        public void deleteAllDeltas() {
            deltasDao.deleteAllForDatabases(
                    uid(),
                    Cf.list(getHandle().handleValue())
            );
        }

        public void saveDeltas(ListF<Delta> deltas) {
            String handle = getHandle().handleValue();
            deltasDao.insertBatch(
                    uid(),
                    deltas.zipWith(delta -> handle)
                            .invert()
            );
        }

        private DatabaseHandle getHandle() {
            return getHandleO()
                    .getOrThrow(userDbRef::consNotFound);
        }

        private Option<DatabaseHandle> getHandleO() {
            return userDbRef.dbHandleO()
                    .orElse(() -> appManager.consPermanentDbHandleO(uid(), databaseRef()))
                    .orElse(() -> getDatabaseO().map(db -> db.dbHandle));
        }

        private void step(String name) {
            DataApiStats.step(name);
        }

        @Override
        public DsSessionTxManager tx() {
            return transactionManager;
        }
    }

    private class SessionTxManager implements DsSessionTxManager {
        private final Session session;

        SessionTxManager(Session session) {
            this.session = session;
        }

        @Override
        public Session session() {
            return session;
        }

        @Override
        public <R> Option<R> executeInTxWithLockedDbIfDbExists(Function<Database, R> function) {
            try {
                return DataApiStats.executeWithProfiling(() ->
                        executeInReadCommittedTx(() ->
                                executeWithDbLockedForUpdateO(function)
                        )
                );
            } catch(NotFoundException e) {
                return Option.empty();
            }
        }

        private <R> Option<R> executeWithDbLockedForUpdateO(Function<Database, R> function) {
            Option<Database> databaseO = session.retrieveDatabaseLockedForUpdateO();
            session.step("find-database");
            return databaseO.map(function::apply);
        }

        @Override
        public <T> T executeInReadCommittedTx(Supplier<T> op) {
            return executeInTx(op, consTxDefinition(TransactionDefinition.ISOLATION_READ_COMMITTED));
        }

        @Override
        public <T> T executeInReadOnlyTx(Supplier<T> op) {
            return executeInTx(op, consReadOnlyTxDefinition(TransactionDefinition.ISOLATION_REPEATABLE_READ));
        }

        private <T> T executeInTx(Supplier<T> op, DefaultTransactionDefinition txDefinition) {
            TransactionStatus transaction;
            try {
                transaction = transactionManager.getTransaction(session.uid(), txDefinition);
                session.step("create-transaction");
            } catch (UserNotFoundException e) {
                throw session.databaseRef().consNotFoundDueToMissing(session.uid());
            }

            try {
                T result = op.get();

                transactionManager.commit(session.uid(), transaction);
                session.step("commit-transaction");

                return result;
            } catch (Throwable e) {
                ExceptionUtils.throwIfUnrecoverable(e);

                logger.debug("Rolling transaction back on exception: {}", ExceptionUtils.getAllMessages(e));
                try {
                    transactionManager.rollback(session.uid(), transaction);
                } catch (Throwable t) {
                    e.addSuppressed(t);
                }
                throw e;
            }
        }

        private DefaultTransactionDefinition consTxDefinition(int isolation) {
            return consTxDefinition(isolation, false);
        }

        private DefaultTransactionDefinition consReadOnlyTxDefinition(int isolation) {
            return consTxDefinition(isolation, true);
        }

        private DefaultTransactionDefinition consTxDefinition(int isolation, boolean readOnly) {
            DefaultTransactionDefinition definition = new DefaultTransactionDefinition();
            definition.setIsolationLevel(isolation);
            definition.setReadOnly(readOnly);
            return definition;
        }
    }
}
