package ru.yandex.chemodan.app.dataapi.core.datasources.disk;

import java.io.File;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Phaser;

import org.joda.time.Duration;
import org.joda.time.Instant;
import org.junit.After;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.transaction.TransactionStatus;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.bolts.function.Function;
import ru.yandex.bolts.function.Function0;
import ru.yandex.bolts.function.Function1V;
import ru.yandex.bolts.function.Function2V;
import ru.yandex.bolts.function.Function3V;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContext;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.CollectionIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DataColumn;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.DataCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordIdColumn;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordIdCondition;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByDataRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByIdRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.protobuf.ProtobufDataUtils;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.DatabaseRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotWithSource;
import ru.yandex.chemodan.app.dataapi.api.datasource.ExtendedDataSource;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.filter.DatabasesFilterSource;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefs;
import ru.yandex.chemodan.app.dataapi.api.db.ref.GlobalDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.SpecialDatabases;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.revision.DatabaseRefRevisions;
import ru.yandex.chemodan.app.dataapi.api.db.revision.DatabaseRevision;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaUtilsTest;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaValidationException;
import ru.yandex.chemodan.app.dataapi.api.deltas.ModifiedRecordsPojo;
import ru.yandex.chemodan.app.dataapi.api.deltas.OutdatedChangeException;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChangeType;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiDeviceIdUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabaseLockMode;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabaseLockedException;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DatabasesJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.support.ShardedTransactionManager;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.datasources.ydb.YdbDataSource;
import ru.yandex.chemodan.app.dataapi.core.dump.DatabaseChangesTskvLogPatternLayout;
import ru.yandex.chemodan.app.dataapi.core.dump.DatabaseChangesTskvLogger;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReference;
import ru.yandex.chemodan.app.dataapi.core.mdssnapshot.MdsSnapshotReferenceJdbcDao;
import ru.yandex.chemodan.app.dataapi.support.RecordField;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.dataapi.test.DataCleaner;
import ru.yandex.chemodan.app.dataapi.test.TestConstants;
import ru.yandex.chemodan.app.dataapi.utils.elliptics.EllipticsHelper;
import ru.yandex.chemodan.app.dataapi.web.AccessForbiddenException;
import ru.yandex.chemodan.log.Log4jHelper;
import ru.yandex.commune.dynproperties.DynamicPropertyManager;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.io.file.File2;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.test.TestDirRule;
import ru.yandex.misc.time.MoscowTime;
import ru.yandex.misc.version.SimpleAppName;

/**
 * @author tolmalev
 */
@ActivateDataApiEmbeddedPg
public class DiskDataSourceTest extends DataApiTestSupport {
    private static final RevisionCheckMode PER_RECORD_CHECK_MODE = RevisionCheckMode.PER_RECORD;

    private static final DatabaseContext GLOBAL_CONTEXT = DatabaseContext.global();

    @ClassRule
    public static final TestDirRule testDir = new TestDirRule();
    private static File logFile;

    protected ExtendedDataSource dataSource;

    @Autowired
    private DataApiManager dataApiManager;
    @Autowired
    protected AppSettingsRegistry cacheSettingsRegistry;
    @Autowired
    private DiskDataSource diskDataSource;
    @Autowired
    private DynamicPropertyManager dynamicPropertyManager;

    @Autowired
    private MdsSnapshotReferenceJdbcDao mdsSnapshotReferenceJdbcDao;

    @Autowired
    private EllipticsHelper ellipticsHelper;

    @Autowired
    private DataCleaner dataCleaner;

    @Autowired
    private ShardedTransactionManager transactionManager;
    @Autowired
    private DatabasesJdbcDao databasesJdbcDao;

    private DataApiUserId uid;

    private UserDatabaseSpec globalDatabaseSpec;

    private final GlobalDatabaseRef globalDatabaseRef = new GlobalDatabaseRef("db_id");

    private final DatabaseHandle databaseHandle = globalDatabaseRef.consHandle("db_id");

    @BeforeClass
    public static void prepareLogger() {
        logFile = new File(testDir.testDir + "/" + "log.txt");
        Log4jHelper.appenderBuilder()
                .appName(new SimpleAppName("dataapi", "dataapi"))
                .name(DatabaseChangesTskvLogger.tskvLogger.name)
                .postfix("-database-changes")
                .layout(new DatabaseChangesTskvLogPatternLayout())
                .fileName(logFile.getAbsolutePath())
                .buffSize(0)
                .build();

    }
    @Before
    public void Before() {
        dataSource = diskDataSource;
        timeStop();
        uid = createRandomCleanUserInDefaultShard();
        globalDatabaseSpec = new UserDatabaseSpec(uid, globalDatabaseRef);
        SpecialDatabases.setMdsSnapshotableAnyForTest(true);

        cacheSettingsRegistry.getDatabaseSettings(new AppDatabaseRef("dataapi", "dataapi")).setHaveDumpInYt(true);
    }

    @After
    public void After() {
        timeStart();
        SpecialDatabases.setMdsSnapshotableAnyForTest(false);
    }

    @Test
    public void getRecordsCount_OneCollection_ReturnsExceptedCount() {
        dataApiManager.createDatabase(globalDatabaseSpec);
        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, makeInsertDelta(10, "col1"));
        dataApiManager.applyDelta(globalDatabaseSpec, 1, PER_RECORD_CHECK_MODE, makeInsertDelta(20, "col1"));
        Assert.equals(2, dataApiManager.getRecordsCount(globalDatabaseSpec,
                RecordsFilter.DEFAULT.withCollectionId("col1"))
        );
    }

    @Test
    public void getRecordsCount_AllDatabase_ReturnsExceptedCount() {
        dataApiManager.createDatabase(globalDatabaseSpec);

        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, makeInsertDelta(10, "col1"));
        dataApiManager.applyDelta(globalDatabaseSpec, 1, PER_RECORD_CHECK_MODE, makeInsertDelta(20, "col2"));
        dataApiManager.applyDelta(globalDatabaseSpec, 2, PER_RECORD_CHECK_MODE, makeInsertDelta(20, "col3"));
        Assert.equals(3, dataApiManager.getRecordsCount(globalDatabaseSpec));
    }

    @Test
    public void getAndInitCopyOnWrite_UnknownUser_ReturnsNothing() {
        Assert.isEmpty(getAndInitCopyOnWrite(new DataApiDeviceIdUserId("dev")));
    }

    @Test
    public void getAndInitCopyOnWrite_DatabaseNotExists_DoNotCreateSnapshotReference() {
        dataCleaner.deleteAllSnapshotReferences(uid);

        Option<Database> dbO = getAndInitCopyOnWrite();
        Assert.isTrue(!dbO.isPresent());

        ListF<MdsSnapshotReference> allReferences = mdsSnapshotReferenceJdbcDao.findAllWithCreationTimeLessThan(
                uid,
                Instant.now().plus(Duration.standardDays(100))
        );
        Assert.isTrue(allReferences.isEmpty());
    }

    @Test
    public void getAndInitCopyOnWrite_DatabaseExists_CreatesSnapshotReferenceWithoutMdsKey() {
        Database createdDb = dataApiManager.createDatabase(globalDatabaseSpec);
        Option<Database> foundDbO = getAndInitCopyOnWrite();
        Assert.equals(createdDb.asExisting(), foundDbO.get());

        Option<MdsSnapshotReference> mdsSnapshotReferenceO = mdsSnapshotReferenceJdbcDao.find(
                uid, createdDb.handleValue(), createdDb.rev);
        Assert.isTrue(mdsSnapshotReferenceO.isPresent());
        Assert.isEmpty(mdsSnapshotReferenceO.get().mdsKey);
    }

    @Test
    public void getAndInitCopyOnWrite_SnapshotWithValidMdsKey_UpdateLastRequestTime() {
        Database database = dataApiManager.createDatabase(globalDatabaseSpec);

        Instant yesterday = now.minus(Duration.standardDays(1));
        Option<String> key = Option.of("key");
        MdsSnapshotReference mdsSnapshotReference = new MdsSnapshotReference(
                database.handleValue(), database.rev, yesterday, key, uid);
        mdsSnapshotReferenceJdbcDao.insert(mdsSnapshotReference);

        getAndInitCopyOnWrite();

        MdsSnapshotReference updatedSnapshotReference =
                mdsSnapshotReferenceJdbcDao.find(uid, database.handleValue(), database.rev).get();
        Assert.equals(now, updatedSnapshotReference.lastRequestTime);
        Assert.equals(key, updatedSnapshotReference.mdsKey);
    }

    @Test
    public void applyDelta_SnapshotReferenceWithoutMdsKeyExists_SetMdsKeyInSnapshotReferenceAndSaveSnapshotToMds() {
        //Arrange
        Database emptyDb = dataApiManager.createDatabase(globalDatabaseSpec);
        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, makeSimpleDelta(100));
        getAndInitCopyOnWrite();

        //Act
        dataApiManager.applyDelta(globalDatabaseSpec, 1, PER_RECORD_CHECK_MODE, makeSimpleDelta(200));

        //Assert
        Option<MdsSnapshotReference> referenceO = mdsSnapshotReferenceJdbcDao.find(uid, emptyDb.handleValue(), 1);
        Assert.some(referenceO, StringUtils.format(
                "Not found mds reference for uid={}, dbRef={}, handle={}",
                emptyDb.uid, emptyDb.dbRef(), emptyDb.handleValue())
        );

        MdsSnapshotReference updatedSnapshotReference = referenceO.get();
        Assert.notEmpty(updatedSnapshotReference.mdsKey);
        Assert.isEmpty(mdsSnapshotReferenceJdbcDao.find(uid, emptyDb.handleValue(), 0));
        Assert.isEmpty(mdsSnapshotReferenceJdbcDao.find(uid, emptyDb.handleValue(), 2));

        String mdsKey = updatedSnapshotReference.mdsKey.get();
        byte[] snapshotBytes = ellipticsHelper.download(mdsKey);
        Snapshot oldSnapshotFromMds = ProtobufDataUtils.snapshotPojoSerializer.deserialize(snapshotBytes)
                .toDomainObject(uid, globalDatabaseRef);

        Assert.equals(1L, oldSnapshotFromMds.database.rev);
        Assert.equals(100L, getValueFromSnapshot(oldSnapshotFromMds));

        Snapshot newSnapshot = dataApiManager.getSnapshot(globalDatabaseSpec);
        Assert.equals(2L, newSnapshot.database.rev);
        Assert.equals(200L, getValueFromSnapshot(newSnapshot));

        testLogs();
    }

    private void testLogs() {
        String content = new File2(logFile).readText();
        Assert.assertContains(content, "method");
        Assert.assertContains(content, "record_data");
        Assert.assertContains(content, "collection_id");
        Assert.assertContains(content, "record_id");
    }

    @Test
    public void applyDelta_PerRecord() {
        dataApiManager.createDatabase(globalDatabaseSpec);

        DataRecordId freshId = new DataRecordId(databaseHandle, "coll", "fresh");
        DataRecordId staleId = new DataRecordId(databaseHandle, "coll", "stale");
        DataRecordId rogueId = new DataRecordId(databaseHandle, "coll", "rogue");

        applyDeltaPerRecord(0, new Delta(Cf.list(
                createEmptyChange(freshId, RecordChangeType.SET),
                createEmptyChange(staleId, RecordChangeType.SET))));

        applyDeltaPerRecord(1, new Delta(createEmptyChange(rogueId, RecordChangeType.INSERT)));

        applyDeltaPerRecord(2, new Delta(Cf.list(
                createEmptyChange(freshId, RecordChangeType.SET),
                createEmptyChange(rogueId, RecordChangeType.DELETE))));

        applyDeltaPerRecord(1, new Delta(Cf.list(
                createEmptyChange(staleId, RecordChangeType.SET),
                createEmptyChange(rogueId, RecordChangeType.INSERT))));

        Snapshot snapshot = dataApiManager.getSnapshot(globalDatabaseSpec);
        Assert.equals(4L, snapshot.database.rev);

        ListF<DataRecordId> actualRecIds = snapshot.records().map(o -> o.id);
        Assert.hasSize(3, actualRecIds);
        SetF<DataRecordId> set = Cf.set(freshId, rogueId, staleId);
        Assert.equals(
                simplifyRecordIds(set),
                simplifyRecordIds(actualRecIds.unique())
        );
    }

    @Test
    public void applyDelta_PerRecordSetForced() {
        Function1V<ListF<Delta>> applyDeltas = deltas -> dataApiManager.applyDeltas(
                globalDatabaseSpec, -1, PER_RECORD_CHECK_MODE, deltas);

        Function1V<ListF<Delta>> verifyOutdated = deltas ->
                Assert.assertThrows(() -> applyDeltas.apply(deltas), OutdatedChangeException.class);

        dataApiManager.createDatabase(globalDatabaseSpec);

        DataRecordId recId = new DataRecordId(databaseHandle, "coll", "rec");

        Delta insert = new Delta(RecordChange.insert(recId, Cf.map()));
        Delta delete = new Delta(RecordChange.delete(recId));
        Delta forced = new Delta(RecordChange.setForced(recId, Cf.map()));
        Delta set = new Delta(RecordChange.set(recId, Cf.map()));

        applyDeltas.apply(Cf.list(forced));

        verifyOutdated.apply(Cf.list(set));
        applyDeltas.apply(Cf.list(forced));

        verifyOutdated.apply(Cf.list(delete, forced));
        applyDeltas.apply(Cf.list(forced, delete, insert));

        Snapshot snapshot = dataApiManager.getSnapshot(globalDatabaseSpec);
        Assert.equals(5L, snapshot.database.rev);

        Assert.equals(1, snapshot.recordCount());
    }

    private ListF<RecordId> simplifyRecordIds(SetF<DataRecordId> recordIds) {
        return recordIds.map(DiskDataSourceTest::simplifyRecordId);
    }

    private static RecordId simplifyRecordId(RecordId recordId) {
        return new SimpleRecordId(recordId.collectionId(), recordId.collectionId());
    }

    @Test
    public void applyDelta_PerRecord_OutdatedRecord() {
        dataApiManager.createDatabase(globalDatabaseSpec);

        DataRecordId recId = new DataRecordId(databaseHandle, "coll", "rec");
        Delta setDelta = new Delta(createEmptyChange(recId, RecordChangeType.SET));

        applyDeltaPerRecord(0, setDelta);
        applyDeltaPerRecord(1, setDelta);

        try {
            applyDeltaPerRecord(0, setDelta);
        } catch (Exception e) {
            Assert.isInstance(e, OutdatedChangeException.class);
        }
    }

    @Test
    public void applyDelta_PerRecord_UpdateNonExisting() {
        dataApiManager.createDatabase(globalDatabaseSpec);

        DataRecordId initialId = new DataRecordId(databaseHandle, "coll", "initial");
        applyDeltaPerRecord(0, new Delta(createEmptyChange(initialId, RecordChangeType.SET)));

        DataRecordId recId = new DataRecordId(databaseHandle, "coll", "rec");
        Delta updateDelta = new Delta(createEmptyChange(recId, RecordChangeType.UPDATE));

        try {
            applyDeltaPerRecord(1, updateDelta);
        } catch (Exception e) {
            Assert.isInstance(e, DeltaValidationException.class);
        }
        try {
            applyDeltaPerRecord(0, updateDelta);
        } catch (Exception e) {
            Assert.isInstance(e, OutdatedChangeException.class);
        }
        try {
            applyDeltaPerRecord(0, new Delta(createEmptyChange(recId, RecordChangeType.SET)));
        } catch (OutdatedChangeException e) {
            Assert.fail("Record set operation should ignore non-existence");
        }
    }

    private Database applyDeltaPerRecord(long rev, Delta delta) {
        return dataApiManager.applyDeltas(globalDatabaseSpec, rev, PER_RECORD_CHECK_MODE, Cf.list(delta)).database;
    }

    private static RecordChange createEmptyChange(DataRecordId recId, RecordChangeType type) {
        return DeltaUtilsTest.createEmptyChange(recId, type);
    }

    @Test
    public void getRevisionSnapshot_RevisionExistsInDb_ReturnsRevisionSnapshotFromDb() {
        Database emptyDb = dataApiManager.createDatabase(globalDatabaseSpec);
        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, makeSimpleDelta(100));

        getSnapshotAndAssert(emptyDb, 1, 100);
    }

    @Test
    public void getRevisionSnapshot_RevisionNotExistsInDb_ReturnsOldRevisionSnapshot() {
        Database emptyDb = dataApiManager.createDatabase(globalDatabaseSpec);
        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, makeSimpleDelta(100));
        getAndInitCopyOnWrite();
        dataApiManager.applyDelta(globalDatabaseSpec, 1, PER_RECORD_CHECK_MODE, makeSimpleDelta(200));

        //old - from MDS
        getSnapshotAndAssert(emptyDb, 1, 100);

        //last - from database
        getSnapshotAndAssert(emptyDb, 2, 200);
    }

    @Test
    public void getRevisionSnapshot_WithSpecificCollection_ReturnsFilteredSnapshot() {
        Database emptyDb = dataApiManager.createDatabase(globalDatabaseSpec);
        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, makeSimpleDelta(10, "col1"));
        dataApiManager.applyDelta(globalDatabaseSpec, 1, PER_RECORD_CHECK_MODE, makeSimpleDelta(20, "col2"));
        dataApiManager.applyDelta(globalDatabaseSpec, 2, PER_RECORD_CHECK_MODE, makeSimpleDelta(30, "col3"));
        getAndInitCopyOnWrite();
        dataApiManager.applyDelta(globalDatabaseSpec, 3, PER_RECORD_CHECK_MODE, makeSimpleDelta(40, "col4"));

        SetF<String> neededCollections = Cf.set("col1", "col3", "col4");
        Snapshot snapshot = dataApiManager.getSnapshotWithRevisionO(
                UserDatabaseSpec.fromDatabase(emptyDb),
                3,
                RecordsFilter.DEFAULT.withCollectionIds(neededCollections)
        ).get().getSnapshot();

        Assert.sizeIs(2, snapshot.records());

        Assert.equals(10L, getValueFromSnapshot(snapshot, "col1"));
        Assert.equals(30L, getValueFromSnapshot(snapshot, "col3"));
    }

    @Test
    public void getRevisionSnapshot_ReverseOrder() {
        dataApiManager.createDatabase(globalDatabaseSpec);
        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, makeSimpleDelta(10, "col1"));
        dataApiManager.applyDelta(globalDatabaseSpec, 1, PER_RECORD_CHECK_MODE, makeSimpleDelta(20, "col2"));
        dataApiManager.applyDelta(globalDatabaseSpec, 2, PER_RECORD_CHECK_MODE, makeSimpleDelta(30, "col3"));
        dataApiManager.applyDelta(globalDatabaseSpec, 3, PER_RECORD_CHECK_MODE, makeSimpleDelta(40, "col4"));

        Database emptyDb = dataApiManager.getDatabase(globalDatabaseSpec);
        Snapshot snapshot = dataApiManager.getSnapshotWithRevisionO(
                UserDatabaseSpec.fromDatabase(emptyDb),
                4,
                RecordsFilter.DEFAULT
                        .withRecordOrder(ByIdRecordOrder.COLLECTION_ID_DESC_RECORD_ID_DESC)
                        .withLimits(SqlLimits.range(1, 2))
        ).get().getSnapshot();

        Assert.sizeIs(2, snapshot.records());
        Assert.equals(20L, getValueFromSnapshot(snapshot, "col2"));
        Assert.equals(30L, getValueFromSnapshot(snapshot, "col3"));
    }

    private static long getValueFromSnapshot(Snapshot revisionSnapshot) {
        return getValueFrom(revisionSnapshot.records());
    }

    private static long getValueFromSnapshot(Snapshot snapshot, String collectionId) {
        return getValueFrom(snapshot.records()
                .filter(row -> row.getCollectionId().equals(collectionId)));
    }

    private static long getValueFrom(CollectionF<DataRecord> records) {
        return records.iterator().next().getData().getTs(TEST_VALUE_NAME).integerValue();
    }

    private void getSnapshotAndAssert(Database database, long revision, long expectedValue) {
        UserDatabaseSpec databaseSpec = new UserDatabaseSpec(uid, database.dbRef());
        database = dataApiManager.getDatabase(databaseSpec);
        Option<SnapshotWithSource> revisionSnapshotO = dataApiManager.getSnapshotWithRevisionO(
                UserDatabaseSpec.fromDatabase(database), revision,
                RecordsFilter.DEFAULT);

        Snapshot revisionSnapshot = revisionSnapshotO.get().getSnapshot();

        Assert.equals(revision, revisionSnapshot.database.rev);
        Assert.sizeIs(1, revisionSnapshot.records());
        Assert.equals(expectedValue, getValueFromSnapshot(revisionSnapshot));

        revisionSnapshotO = dataApiManager.getSnapshotWithRevisionO(
                UserDatabaseSpec.fromDatabase(database), revision,
                RecordsFilter.DEFAULT.withRecordOrder(ByIdRecordOrder.COLLECTION_ID_DESC_RECORD_ID_DESC)
        );

        revisionSnapshot = revisionSnapshotO.get().getSnapshot();

        Assert.equals(revision, revisionSnapshot.database.rev);
        Assert.sizeIs(1, revisionSnapshot.records());
        Assert.equals(expectedValue, getValueFromSnapshot(revisionSnapshot));
    }

    @Test
    public void getSnapshotTest() {
        dataApiManager.createDatabase(globalDatabaseSpec);
        Snapshot snapshot = dataApiManager.getSnapshot(globalDatabaseSpec);
        Assert.equals(0L, snapshot.database.rev);
    }

    @Test
    public void getRecordsModifiedSinceRevisions() {
        String db1Id = "db_1";
        String db2Id = "db_2";

        ListF<String> db1RecordIds = Cf.list("db_1_rec_1", "db_1_rec_2", "db_1_rec_3");
        ListF<String> db2RecordIds = Cf.list("db_2_rec_1", "db_2_rec_2");

        Function2V<String, ListF<String>> createDatabaseWithRecordsF = (dbId, recIds) -> {
            GlobalDatabaseRef dbRef = new GlobalDatabaseRef(dbId);
            UserDatabaseSpec databaseSpec = new UserDatabaseSpec(uid, dbRef);
            dataApiManager.createDatabase(databaseSpec);
            ListF<Delta> deltas = recIds.map(
                    id -> new Delta(createEmptyChange(
                            new DataRecordId(dbRef.consHandle(db1Id), "col", id), RecordChangeType.INSERT)
                    )
            );
            dataApiManager.applyDeltas(databaseSpec, 0, PER_RECORD_CHECK_MODE, deltas);
        };
        createDatabaseWithRecordsF.apply(db1Id, db1RecordIds);
        createDatabaseWithRecordsF.apply(db2Id, db2RecordIds);

        ModifiedRecordsPojo records;

        records = dataSource.getRecordsModifiedSinceRevisions(
                uid,
                new DatabaseRefRevisions(GLOBAL_CONTEXT, Cf.list(
                        new DatabaseRevision(db1Id, 0),
                        new DatabaseRevision(db2Id, 0))
                ),
                Option.of(2)
        );

        Assert.hasSize(2, records.objects);
        Assert.equals(
                Cf.set(db1RecordIds.first(), db2RecordIds.first()),
                records.objects.map(DatabaseRecord::getRecordId).unique());
        Assert.isTrue(records.hasMore);

        records = dataSource.getRecordsModifiedSinceRevisions(
                uid,
                new DatabaseRefRevisions(GLOBAL_CONTEXT, records.progressRevisions),
                Option.of(3)
        );

        Assert.hasSize(3, records.objects);
        Assert.equals(
                db1RecordIds.drop(1).unique().plus(db2RecordIds.drop(1)),
                records.objects.map(DatabaseRecord::getRecordId).unique());
        Assert.isFalse(records.hasMore);

        records = dataSource.getRecordsModifiedSinceRevisions(
                uid,
                new DatabaseRefRevisions(GLOBAL_CONTEXT, Cf.list(
                        new DatabaseRevision(db1Id, 0),
                        new DatabaseRevision(db2Id, 100500))
                ),
                Option.empty()
        );

        Assert.equals(db1RecordIds, records.objects.map(DatabaseRecord::getRecordId));
        Assert.isFalse(records.hasMore);

        records = dataSource.getRecordsModifiedSinceRevisions(
                uid,
                new DatabaseRefRevisions(GLOBAL_CONTEXT, Cf.list(
                        new DatabaseRevision(db2Id, 100500),
                        new DatabaseRevision("unknown", 100500))
                ),
                Option.empty()
        );

        Assert.isEmpty(records.objects);
        Assert.equals(Cf.list(100500L, 100500L), records.progressRevisions.map(DatabaseRevision::getRev));
        Assert.isFalse(records.hasMore);
    }

    @Test
    public void getRecordsByCollectionIdCondition() {
        Database db = dataApiManager.createDatabase(globalDatabaseSpec);

        Delta delta = new Delta(
                Cf.range(0, 3).map(num -> createEmptyChange(
                        new DataRecordId(db.dbHandle, "col" + (num > 0 ? "_" + num : ""), "rec"),
                        RecordChangeType.INSERT)
                )
        );
        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, delta);

        Function<CollectionIdCondition, ListF<String>> find = cond -> {
            ListF<DataRecord> records = dataApiManager.getRecords(db.spec(), RecordsFilter.DEFAULT.withCollectionIdCond(cond));
            Assert.forAll(records, r -> cond.matches(r.getCollectionId()));

            return records.map(DataRecord::getCollectionId);
        };

        Assert.hasSize(3, find.apply(CollectionIdCondition.all()));
        Assert.hasSize(3, find.apply(CollectionIdCondition.like("col%")));

        Assert.equals(Cf.list("col"), find.apply(CollectionIdCondition.eq("col")));
        Assert.equals(Cf.list("col_1", "col_2"), find.apply(CollectionIdCondition.inSet(Cf.list("col_1", "col_2"))));
        Assert.equals(Cf.list("col_1"), find.apply(CollectionIdCondition.notInSet(Cf.list("col", "col_2"))));
    }

    @Test
    public void getRecordsByRecordIdCondition() {
        Database db = dataApiManager.createDatabase(globalDatabaseSpec);

        Delta delta = new Delta(
                Cf.range(0, 3).map(num -> createEmptyChange(
                        new DataRecordId(db.dbHandle, "col", "rec" + (num > 0 ? "_" + num : "")),
                        RecordChangeType.INSERT)
                )
        );
        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, delta);

        Function<RecordIdCondition, ListF<String>> find = cond -> {
            ListF<DataRecord> records = dataApiManager.getRecords(db.spec(), RecordsFilter.DEFAULT.withRecordIdCond(cond));
            Assert.forAll(records, r -> cond.matches(r.getRecordId()));

            return records.map(DataRecord::getRecordId);
        };

        Assert.hasSize(3, find.apply(RecordIdCondition.all()));
        Assert.equals(Cf.list("rec", "rec_1"), find.apply(RecordIdCondition.between("rec", "rec_1")));
    }

    @Test
    public void getRecordsByDataCondition() {
        Database db = dataApiManager.createDatabase(globalDatabaseSpec);
        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, new Delta(Cf.list(
                RecordChange.insert("coll", "record_ab", Cf.toMap(Tuple2List.fromPairs(
                        "field1", DataField.string("A"),
                        "field2", DataField.string("B"),
                        "code", DataField.string("ab"),
                        "decimal", DataField.decimal(6.75),
                        "ts", DataField.timestamp(new Instant(0))))),

                RecordChange.insert("coll", "record_ac", Cf.toMap(Tuple2List.fromPairs(
                        "field1", DataField.string("A"),
                        "field2", DataField.string("C"),
                        "code", DataField.string("ac"),
                        "bool", DataField.bool(true),
                        "ts", DataField.timestamp(new Instant(3))))),

                RecordChange.insert("coll", "record_bc", Cf.toMap(Tuple2List.fromPairs(
                        "field1", DataField.string("B"),
                        "field2", DataField.string("C"),
                        "code", DataField.string("bc"),
                        "bool", DataField.bool(false),
                        "decimal", DataField.decimal(5.5),
                        "ts", DataField.timestamp(new Instant(11))))))));

        Function<RecordCondition, ListF<String>> find = cond -> {
            ListF<DataRecord> records = dataApiManager.getRecords(db.spec(), RecordsFilter.DEFAULT.withRecordCond(cond));

            Assert.equals(dataApiManager.getRecords(db.spec(), RecordsFilter.DEFAULT).filter(cond::matches), records);

            return records.map(DataRecord::getRecordId);
        };

        DataColumn<String> field1 = DataColumn.string("field1");
        DataColumn<String> field2 = DataColumn.string("field2");

        DataColumn<String> code = DataColumn.string("code");
        DataColumn<Boolean> bool = DataColumn.bool("bool");

        DataColumn<Double> decimal = DataColumn.decimal("decimal");
        DataColumn<Instant> ts = DataColumn.timestamp("ts");

        RecordIdColumn id = RecordIdColumn.C;

        Assert.hasSize(3, find.apply(DataCondition.all()));

        Assert.equals(Cf.list("record_ac", "record_bc"), find.apply(field2.eq("C")));
        Assert.equals(Cf.list("record_ac", "record_bc"), find.apply(ts.ge(new Instant(2))));

        Assert.equals(Cf.list("record_ac"), find.apply(field1.eq("A").and(field2.eq("C"))));

        Assert.equals(Cf.list("record_ab", "record_ac"), find.apply(code.startsWith("a")));
        Assert.equals(Cf.list("record_bc"), find.apply(bool.startsWith("fa")));

        Assert.equals(Cf.list("record_ab"), find.apply(decimal.gt(5.5)));
        Assert.equals(Cf.list("record_bc"), find.apply(decimal.gt(5.5).not()));
        Assert.equals(Cf.list("record_ab", "record_bc"), find.apply(decimal.between(0d, 100d)));

        Assert.equals(Cf.list("record_ac", "record_bc"), find.apply(id.inSet(Cf.list("record_ac", "record_bc"))));
        Assert.equals(Cf.list("record_ac"), find.apply(field1.eq("A").and(id.inSet(Cf.list("record_ac")))));

        Assert.equals(Cf.list(), find.apply(bool.eq(false).or(id.eq("record_ac")).not()));
        Assert.equals(Cf.list("record_ac", "record_bc"), find.apply(bool.eq(true).and(id.eq("record_ab")).not()));

        Assert.equals(Cf.list("record_ab"), find.apply(bool.isNull()));
        Assert.equals(Cf.list("record_ac", "record_bc"), find.apply(bool.isNotNull()));
    }

    @Test
    public void getRecordsOrderedByDataField() {
        Database db = dataApiManager.createDatabase(globalDatabaseSpec);
        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, new Delta(Cf.list(
                RecordChange.insert("coll", "rec1", Cf.map(
                        "str", DataField.string("rec1"),
                        "ts", DataField.timestamp(new Instant(0)),
                        "num", DataField.integer(11))),

                RecordChange.insert("coll", "rec2", Cf.map(
                        "ts", DataField.timestamp(new Instant(1)),
                        "num", DataField.integer(2))),

                RecordChange.insert("coll", "rec3", Cf.map(
                        "str", DataField.string("rec3"),
                        "ts", DataField.timestamp(new Instant(2)))))));

        Function<ByDataRecordOrder, ListF<String>> find = order -> {
            ListF<DataRecord> records = dataApiManager.getRecords(db.spec(), RecordsFilter.DEFAULT.withRecordOrder(order));

            Assert.equals(records.map(DataRecord::getRecordId),
                    records.reverse().sorted(order.comparator()).map(DataRecord::getRecordId));

            return records.map(DataRecord::getRecordId);
        };

        Assert.equals(Cf.list("rec3", "rec2", "rec1"), find.apply(RecordField.instant("ts").column().orderByDesc()));
        Assert.equals(Cf.list("rec2", "rec1", "rec3"), find.apply(RecordField.intNumber("num").column().orderBy()));
        Assert.equals(Cf.list("rec1", "rec3", "rec2"), find.apply(RecordField.string("str").column().orderBy()));
    }

    @Test
    public void createDatabase() {
        AppDatabaseRef databaseRef = new AppDatabaseRef("app", "database-1");
        UserDatabaseSpec databaseSpec = new UserDatabaseSpec(uid, databaseRef);

        Assert.isEmpty(dataApiManager.getDatabaseO(databaseSpec));
        Database database = dataApiManager.createDatabase(databaseSpec);

        Assert.equals(0L, database.rev);
        Assert.equals(0L, database.meta.recordsCount);
        Assert.none(database.meta.description);

        Assert.equals(uid, database.uid);
        Assert.equals(databaseRef.databaseId(), database.databaseId());
        Assert.equals(databaseRef.appNameO(), database.appNameO());

        Assert.equals(database.asExisting(), dataApiManager.getDatabase(databaseSpec));
    }

    @Test
    public void createGlobalDatabase() {
        Assert.isEmpty(dataApiManager.getDatabaseO(globalDatabaseSpec));
        Database database = dataApiManager.createDatabase(globalDatabaseSpec);

        Assert.equals(0L, database.rev);
        Assert.equals(0L, database.meta.recordsCount);
        Assert.none(database.meta.description);

        Assert.equals(uid, database.uid);
        Assert.equals(globalDatabaseRef.databaseId(), database.databaseId());
        Assert.equals(globalDatabaseRef.appNameO(), database.appNameO());

        Assert.equals(database.asExisting(), dataApiManager.getDatabase(globalDatabaseSpec));
    }

    @Test(expected = AccessForbiddenException.class)
    public void createExternalDatabaseForbidden() {
        dataApiManager.createDatabase(
                UserDatabaseSpec.fromUserAndAlias(uid, new ExternalDatabaseAlias("client-app", "app", "database-1"))
        );
    }

    @Test(expected = AccessForbiddenException.class)
    public void createExternalDatabaseForbiddenReadOnly() {
        dataApiManager.createDatabase(UserDatabaseSpec.fromUserAndAlias(uid, TestConstants.EXT_DB_ALIAS_RO));
    }

    @Test
    public void createExternalDatabase() {
        UserDatabaseSpec databaseSpec = UserDatabaseSpec.fromUserAndAlias(uid, TestConstants.EXT_DB_ALIAS_RW);

        Database database = dataApiManager.createDatabase(databaseSpec);

        Assert.equals(0L, database.rev);
        Assert.equals(0L, database.meta.recordsCount);
        Assert.none(database.meta.description);

        Assert.equals(uid, database.uid);
        Assert.equals(TestConstants.EXT_DB_ALIAS_RW, database.alias);

        Assert.equals(database.asExisting(), dataApiManager.getDatabase(databaseSpec));
    }

    @Test
    public void completelyRemoveDatabasesDeletedBefore() {
        DataRecordId recId = new DataRecordId(databaseHandle, "coll", "rec");
        Delta setDelta = new Delta(createEmptyChange(recId, RecordChangeType.SET));

        Instant deletionTime = MoscowTime.instant(1986, 11, 20, 3, 0);

        Database filledDb = dataApiManager.createDatabase(globalDatabaseSpec);
        dataApiManager.applyDelta(globalDatabaseSpec, 0, PER_RECORD_CHECK_MODE, setDelta);

        moveDatabasesToDeleted(Cf.list(filledDb), deletionTime);

        Assert.some(dataSource.getDeletedDatabaseO(uid, filledDb.dbHandle));
        Assert.notEmpty(dataApiManager.getRecords(filledDb.spec()));
        Assert.notEmpty(dataApiManager.listDeltas(filledDb.spec(), 0, 1));

        Database emptyDb = dataApiManager.createDatabase(globalDatabaseSpec);
        moveDatabasesToDeleted(Cf.list(emptyDb), deletionTime);

        Assert.some(dataSource.getDeletedDatabaseO(uid, emptyDb.dbHandle));

        dataSource.completelyRemoveDatabasesDeletedBefore(deletionTime.plus(Duration.standardMinutes(1)));

        Assert.none(dataSource.getDeletedDatabaseO(uid, filledDb.dbHandle));
        Assert.isEmpty(dataApiManager.getRecords(filledDb.spec()));
        Assert.isEmpty(dataApiManager.listDeltas(filledDb.spec(), 0, 1));

        Assert.none(dataSource.getDeletedDatabaseO(uid, emptyDb.dbHandle));
    }

    private void moveDatabasesToDeleted(ListF<Database> databases, Instant deletionTime) {
        if (dataSource instanceof DiskDataSource) {
            ((DiskDataSource) dataSource).moveDatabasesToDeleted(databases, deletionTime);
        } else if (dataSource instanceof YdbDataSource) {
            ((YdbDataSource) dataSource).moveDatabasesToDeleted(databases, deletionTime);
        } else {
            throw new IllegalStateException("Unknown data source type " + dataSource.getClass());
        }
    }

    @Test
    public void deleteDatabases() {
        ListF<String> dbIds = Cf.list("db1", "db2");

        Function0<ListF<Database>> createDatabasesF = () -> dbIds.map(id -> {
            GlobalDatabaseRef dbRef = new GlobalDatabaseRef(id);
            Database db = dataApiManager.createDatabase(new UserDatabaseSpec(uid, dbRef));

            DataRecordId recId = new DataRecordId(dbRef.consHandle(id), "coll", "rec");
            Delta delta = new Delta(createEmptyChange(recId, RecordChangeType.SET));

            return dataApiManager.applyDelta(db, PER_RECORD_CHECK_MODE, delta);
        });

        Function3V<DatabaseDeletionMode, Boolean, Integer> testF = (mode, some, expectedDeletedCount) -> {
            createDatabasesF.apply();

            DatabasesFilterSource filterSrc = some ? DatabaseRefs.global(dbIds) : DatabaseContext.global();
            dataSource.deleteDatabases(uid, filterSrc, mode);

            Assert.isEmpty(dataSource.listDatabases(uid));
            Assert.hasSize(expectedDeletedCount, dataSource.listDeletedDatabases(uid));
        };

        testF.apply(DatabaseDeletionMode.REMOVE_COMPLETELY, true, 0);
        testF.apply(DatabaseDeletionMode.REMOVE_COMPLETELY, false, 0);
        testF.apply(DatabaseDeletionMode.MARK_DELETED, true, 2);
        testF.apply(DatabaseDeletionMode.MARK_DELETED, false, 4);
    }

    @Test
    public void nowaitLocking() {
        Database db = dataApiManager.createDatabase(new UserDatabaseSpec(uid, new GlobalDatabaseRef("database")));

        TransactionStatus tx = transactionManager.getTransaction(db.uid);
        try {
            databasesJdbcDao.find(db.uid, db.dbRef(), DatabaseLockMode.FOR_UPDATE);

            CompletableFuture<Void> future = CompletableFuture.runAsync(() -> dataApiManager.applyDelta(
                    db.withNowaitLock(true), RevisionCheckMode.PER_RECORD, new Delta(Cf.list())));

            Assert.assertThrows(future::get, ExecutionException.class, e -> e.getCause() instanceof DatabaseLockedException);

        } finally {
            transactionManager.rollback(db.uid, tx);
        }
    }

    @Test
    public void runWithLockedDatabase() {
        Database db = dataApiManager.createDatabase(new UserDatabaseSpec(uid, new GlobalDatabaseRef("locked")));
        UserDatabaseSpec dbRef = new UserDatabaseSpec(db.uid, db.dbRef());

        Phaser phaser = new Phaser(2);

        CompletableFuture<Void> commitFuture = CompletableFuture.runAsync(() ->
                dataApiManager.runWithLockedDatabase(db.spec(), session -> {
                    phaser.arrive();
                    phaser.arriveAndAwaitAdvance();

                    session.applyDeltas(RevisionCheckMode.PER_RECORD, Cf.list(new Delta(Cf.list())));

                    phaser.arrive();
                    phaser.arriveAndAwaitAdvance();
                }));

        commitFuture.exceptionally(t -> { phaser.arriveAndDeregister(); return null; });

        phaser.arriveAndAwaitAdvance();

        Assert.assertThrows(() -> dataApiManager.applyDelta(db.withNowaitLock(true),
                RevisionCheckMode.PER_RECORD, new Delta(Cf.list())), DatabaseLockedException.class);

        phaser.arrive();
        phaser.arriveAndAwaitAdvance();

        Assert.equals(db.rev, dataApiManager.getDatabase(dbRef).rev, "Unexpected commit");

        phaser.arrive();
        commitFuture.join();

        Assert.equals(db.rev + 1, dataApiManager.getDatabase(dbRef).rev);
    }

    private Option<Database> getAndInitCopyOnWrite() {
        return getAndInitCopyOnWrite(uid);
    }

    private Option<Database> getAndInitCopyOnWrite(DataApiUserId uid) {
        return dataApiManager.getAndInitCopyOnWrite(new UserDatabaseSpec(uid, globalDatabaseRef));
    }
}
