package ru.yandex.chemodan.app.dataapi.core.datasources.migration;

import java.util.UUID;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceType;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.MetaUser;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.PassportDataSource;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.PassportSession;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class DiskToPassportMigration implements DsMigration {
    private static final int DELTA_BATCH_SIZE = 10000;

    private final DiskDataSource diskDs;

    private final PassportDataSource passportDs;

    public DiskToPassportMigration(DiskDataSource diskDs, PassportDataSource passportDs) {
        this.diskDs = diskDs;
        this.passportDs = passportDs;
    }

    @Override
    public Tuple2<DataSourceType, DataSourceType> type() {
        return new Tuple2<>(DataSourceType.DISK, DataSourceType.PASSPORT);
    }

    @Override
    public void migrate(MetaUser user, DatabaseRef databaseRef) {
        UserDatabaseSpec dbSpec = new UserDatabaseSpec(user.getUserId(), databaseRef);
        DiskDataSource.Session diskSession = diskDs.openSession(dbSpec);
        diskSession.tx()
                .runInTxWithLockedDbIfDbExists(
                        db -> new UserMigration(db, diskSession, passportDs.openSession(dbSpec))
                                .migrate()
                );

    }

    private static class UserMigration {
        final Database db;

        final DiskDataSource.Session diskSession;

        final PassportSession passportSession;

        UserMigration(Database db, DiskDataSource.Session diskSession, PassportSession passportSession) {
            this.db = db;
            this.diskSession = diskSession;
            this.passportSession = passportSession;
        }

        void migrate() {
            migrateRecords();
            ListF<Delta> deltas = migrateDeltas();

            diskSession.deleteAllRecords();
            diskSession.deleteAllDeltas();
            diskSession.saveDeltas(deltas);
        }

        private void migrateRecords() {
            ListF<DataRecord> diskRecords = diskSession.getDataRecords(RecordsFilter.DEFAULT);
            saveRecordsToPassport(diskRecords);

            ListF<DataRecord> passportRecords = passportSession.getDataRecords(RecordsFilter.DEFAULT);
            if (!diskRecords.equals(passportRecords)) {
                throw consException("disk snapshot doesn't equal passport snapshot");
            }
        }

        private ListF<Delta> migrateDeltas() {
            Tuple2List<String, Delta> deltas = Tuple2List.tuple2List();
            for(long rev = 0; deltas.size() % DELTA_BATCH_SIZE == 0; rev = deltas.last().get2().getRevUnsafe() + 1) {
                Tuple2List<String, Delta> batchDeltas =
                        diskSession.listDeltas(rev, DELTA_BATCH_SIZE)
                                .sortedBy(Delta::getRevUnsafe)
                                .zipWith(delta -> UUID.randomUUID().toString())
                                .invert();
                saveDeltasToPassport(batchDeltas);
                deltas = deltas.plus(batchDeltas);
            }

            ListF<Tuple2List<String, Delta>> paginatedDeltas = deltas.paginate(DELTA_BATCH_SIZE)
                    .map(Tuple2List::tuple2List);
            for(Tuple2List<String, Delta> batchDeltas : paginatedDeltas) {
                ListF<Delta> passportDeltas = passportSession.listDeltas(batchDeltas.get1())
                        .sortedBy(Delta::getRevUnsafe);
                if (!passportDeltas.equals(batchDeltas.get2())) {
                    throw consException("disk deltas doesn't equal passport deltas");
                }
            }

            return deltas.map((id, delta) -> delta.withId(id).withoutData());
        }

        private void saveRecordsToPassport(ListF<DataRecord> records) {
            passportSession.saveRecordsToPassport(records, db.rev, Option.of(db.rev));
        }

        private void saveDeltasToPassport(Tuple2List<String, Delta> deltasWithIds) {
            passportSession.saveDeltasToPassport(deltasWithIds, db.rev, Option.of(db.rev));
        }
    }

    private static RuntimeException consException(String message) {
        return new IllegalStateException("Error while migrating records: " + message);
    }
}
