package ru.yandex.chemodan.app.dataapi.core.datasources.migration;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import net.jodah.failsafe.FailsafeException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceType;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.MetaUser;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.migration.MigrationSupport;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.PassportDataSource;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.PassportSession;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.errors.PassportUserNotFoundException;
import ru.yandex.chemodan.util.sharpei.ShardUserInfo;
import ru.yandex.chemodan.util.sharpei.SharpeiCachingManager;
import ru.yandex.misc.ExceptionUtils;

@RequiredArgsConstructor
public class PassportToDiskMigration implements DsMigration {

    private final UserMetaManager userMetaManager;
    private final DiskDataSource diskDs;
    private final PassportDataSource passportDs;

    @Override
    public Tuple2<DataSourceType, DataSourceType> type() {
        return new Tuple2<>(DataSourceType.PASSPORT, DataSourceType.DISK);
    }

    @Override
    public void migrate(MetaUser user, DatabaseRef databaseRef) {
        if (!user.isMigrated(databaseRef) && user.isRo()) {
            MigrationSupport.doWithShard(user.getUserId(), user.getShardId(), () -> {
                UserDatabaseSpec spec = new UserDatabaseSpec(user.getUserId(), databaseRef);
                DiskDataSource.Session ds = diskDs.openSession(spec);
                PassportSession ps = passportDs.openSession(spec);
                ps.tx().runInTxWithLockedDbIfDbExists(db -> {
                    Snapshot passportSnapshot = ps.getSnapshotO(RecordsFilter.DEFAULT).get();
                    ds.deleteAllRecords();
                    ds.deleteAllDeltas();
                    ds.save(new DatabaseChange(Snapshot.empty(db), passportSnapshot, Cf.list()));
                    Snapshot diskSnapshot = ds.getSnapshotO(RecordsFilter.DEFAULT).get();
                    if (diskSnapshot.equals(passportSnapshot)) {
                        userMetaManager.updateMigrated(user.getUserId(), databaseRef, true);
                    } else {
                        throw new IllegalStateException("disk snapshot isn't equal to passport snapshot for " + user.getUserId());
                    }
                });
            });
        }
        userMetaManager.updateReadOnly(user.getUserId(), false);
    }

    public Helper helper(String ctx, String db) {
        return new Helper(DatabaseRef.cons(Option.of(ctx), db));
    }

    @RequiredArgsConstructor
    public class Helper {

        private final SetF<DataApiUserId> notFound = Cf.hashSet();
        private final SetF<DataApiUserId> migrated = Cf.hashSet();
        private final SetF<DataApiUserId> readonly = Cf.hashSet();
        private final DatabaseRef ref;
        private boolean stop = false;
        private CompletableFuture<Void> future = null;

        public synchronized void start(int batch) {
            if (!isRunning()) {
                stop = false;
                future = CompletableFuture.runAsync(() -> {
                    while (!isStopping()) {
                        unsetReadOnly();
                        List<DataApiUserId> users = diskDs
                                .getDatabaseUsersStream(ref)
                                .filter(this::shouldMigrate)
                                .limit(batch)
                                .collect(Collectors.toList());
                        if (users.isEmpty()) {
                            break;
                        }
                        synchronized (this) {
                            readonly.addAll(users);
                        }
                        users.parallelStream().forEach(this::setReadOnly);
                        sleep();
                        users.parallelStream().forEach(this::migrate);
                    }
                });
            }
        }

        private synchronized boolean shouldMigrate(DataApiUserId uid) {
            if (migrated.containsTs(uid) || notFound.containsTs(uid)) {
                return false;
            } else {
                if (userMetaManager.findMetaUser(uid).isMatch(u -> u.isMigrated(ref))) {
                    migrated.add(uid);
                    return false;
                } else {
                    return true;
                }
            }
        }

        private void setReadOnly(DataApiUserId uid) {
            userMetaManager.updateReadOnly(uid, true);
        }

        private synchronized Option<DataApiUserId> getReadonly() {
            return readonly.iterator().nextO();
        }

        public void unsetReadOnly() {
            while (true) {
                Option<DataApiUserId> uid = getReadonly();
                uid.ifPresent(u -> {
                    if (userMetaManager.findMetaUser(u).isMatch(ShardUserInfo::isRo)) {
                        userMetaManager.updateReadOnly(u, false);
                    } else {
                        synchronized (this) {
                            readonly.removeTs(u);
                        }
                    }
                });
                if (!uid.isPresent()) {
                    break;
                }
            }
        }

        public long unsetAllReadOnly() {
            return diskDs
                    .getDatabaseUsersStream(ref)
                    .filter(u -> userMetaManager.findMetaUser(u).isMatch(ShardUserInfo::isRo))
                    .peek(u -> userMetaManager.updateReadOnly(u, false))
                    .count();
        }

        private void migrate(DataApiUserId uid) {
            try {
                PassportToDiskMigration.this.migrate(userMetaManager.findMetaUser(uid).get(), ref);
            } catch (PassportUserNotFoundException e) {
                setNotFound(uid);
            } catch (FailsafeException e) {
                if (e.getCause() instanceof PassportUserNotFoundException) {
                    setNotFound(uid);
                } else {
                    throw ExceptionUtils.translate(e);
                }
            }
        }

        @SneakyThrows
        private void sleep() {
            Thread.sleep(2 * SharpeiCachingManager.ttl.get() * 60_000L);
        }

        public synchronized void stop() {
            stop = true;
        }

        public synchronized boolean isRunning() {
            return future != null && !future.isDone();
        }

        public synchronized boolean isStopping() {
            return isRunning() && stop;
        }

        public SetF<DataApiUserId> getMigrated() {
            return migrated.unmodifiable();
        }

        public SetF<DataApiUserId> getNotFound() {
            return notFound.unmodifiable();
        }

        private synchronized void setNotFound(DataApiUserId uid) {
            notFound.add(uid);
        }

        @Override
        public String toString() {
            return status();
        }

        public synchronized String status() {
            return Stream.of(
                    isRunning() ? "running" : "",
                    isStopping() ? "stopping" : "",
                    "done:" + migrated.size(),
                    "404:" + notFound.size(),
                    future == null ? "" : future.toString()
            ).filter(x -> !x.isEmpty()).collect(Collectors.joining(" "));
        }

        public synchronized void purge() {
            notFound.clear();
            migrated.clear();
        }

        public synchronized void clearNotFound() {
            notFound.clear();
        }

    }

}
