package ru.yandex.chemodan.app.dataapi.core.datasources.passport;

import java.util.List;
import java.util.function.Supplier;

import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.RetryPolicy;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceSession;
import ru.yandex.chemodan.app.dataapi.api.datasource.DsSessionTxManager;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseExistsException;
import ru.yandex.chemodan.app.dataapi.api.db.StuckBehindDatabaseException;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaValidationException;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.PassportDataSyncChange;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.PassportDataSyncClient;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.PassportDataSyncData;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.errors.PassportDataSyncRevisionMismatchException;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.request.PassportDataSyncGetDeltas;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.request.PassportDataSyncGetObjects;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.request.PassportDataSyncRequestParam;
import ru.yandex.chemodan.app.dataapi.web.DatabaseNotFoundException;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.misc.db.masterSlave.MasterSlaveContextHolder;
import ru.yandex.misc.db.masterSlave.MasterSlavePolicy;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
@SuppressWarnings("unused")
public class PassportSession implements DataSourceSession {
    private static final int BLACKBOX_DB_RECORD_LIMIT = 1000;
    private static final Logger logger = LoggerFactory.getLogger(PassportSession.class);

    private final DiskDataSource.Session session;

    private final DataApiPassportUserId passportUid;

    private final PassportDataSyncClient passportClient;

    private final UserMetaManager userMetaManager;

    private final RetryPolicy retryPolicy;

    public PassportSession(DiskDataSource.Session session, PassportDataSyncClient passportClient,
            UserMetaManager userMetaManager, RetryPolicy retryPolicy)
    {
        this.session = session;
        this.passportUid = toPassportUid(session.uid());
        this.passportClient = passportClient;
        this.userMetaManager = userMetaManager;
        this.retryPolicy = retryPolicy;
    }

    private static DataApiPassportUserId toPassportUid(DataApiUserId uid) {
        if (!uid.isPassportUid()) {
            throw new IllegalStateException("Expecting Passport UID, got - " + uid);
        }
        return (DataApiPassportUserId) uid;
    }

    @Override
    public DsSessionTxManager tx() {
        return session.tx();
    }

    @Override
    public UserDatabaseSpec databaseSpec() {
        return session.databaseSpec();
    }

    @Override
    public Database getOrCreateDatabase() {
        Option<Database> databaseO = session.getDatabaseO();
        if (databaseO.isPresent()) {
            return databaseO.get();
        }

        try {
            return executeInReadCommittedRegisterUser(() -> {
                Database database = session.createDatabase();
                setRevAtPassportForNewDb(database);
                return database;
            });
        } catch (DatabaseExistsException e) {
            logger.warn("Conflict on database creation - try to find existing");
            return MasterSlaveContextHolder.withPolicy(MasterSlavePolicy.R_M, session::retrieveDatabaseWithoutLockingO)
                    .getOrThrow(session.databaseRef()::consNotFound);
        }
    }

    @Override
    public Database createDatabase() {
        return executeInReadCommittedRegisterUser(this::doCreateDatabase);
    }

    @Override
    public Option<Database> getDatabaseO() {
        return session.getDatabaseO();
    }

    private Database doCreateDatabase() {
        Database database = session.createDatabase();
        setRevAtPassportForNewDb(database);
        return database;
    }

    private Database executeInReadCommittedRegisterUser(Supplier<Database> supplier) {
        try {
            return tx().executeInReadCommittedTx(supplier);
        } catch (DatabaseNotFoundException ex) {
            if (ex.dueToUser) {
                userMetaManager.registerIfNotExists(uid());
                return tx().executeInReadCommittedTx(supplier);
            }

            throw ex;
        }
    }

    private void setRevAtPassportForNewDb(Database database) {
        PassportDataSyncChange change = new PassportDataSyncChange(databaseRef(), database.rev);
        try {
            passportClient.save(passportUid.uid, change);
        } catch (PassportDataSyncRevisionMismatchException ex) {
            passportClient.save(passportUid.uid,
                    change.withOldRevisionO(
                            passportClient.getCurrentRevO(passportUid, databaseRef())
                    )
            );
        }
    }

    // deltas

    @Override
    public Delta getDelta(long rev) {
        return getDelta(session.getDelta(rev).getIdUnsafe());
    }

    private Delta getDelta(String deltaId) {
        return getDataFromBlackbox(new PassportDataSyncGetDeltas(deltaId))
                .getDelta();
    }

    private PassportDataSyncData getDataFromBlackbox(PassportDataSyncRequestParam requestParam) {
        return passportClient.getData(passportUid, requestParam);
    }

    private PassportDataSyncData getDataFromBlackbox(PassportDataSyncGetObjects requestParam, Option<Long> rev) {
        if (!rev.isPresent()) {
            return getDataFromBlackbox(requestParam);
        }

        // Blackbox read operations executed from slaves, therefore delay is possible.
        // So we make several retries in attempt to deal with it.
        return Failsafe.with(retryPolicy.retryOn(throwable -> blackboxIsPossiblyLate(rev.get(), throwable)))
                .get(() -> getDataFromBlackbox(requestParam));
    }

    private boolean blackboxIsPossiblyLate(long requestRev, Throwable throwable) {
        if (!(throwable instanceof PassportDataSyncRevisionMismatchException)) {
            return false;
        }

        PassportDataSyncRevisionMismatchException ex = (PassportDataSyncRevisionMismatchException) throwable;
        return !ex.passportRev.isMatch(passportRev -> passportRev >= requestRev);
    }

    @Override
    public ListF<Delta> listDeltas(long fromRev, int limit) {
        return listDeltas(
                session.listDeltas(fromRev, limit)
                        .map(Delta::getIdUnsafe)
        );
    }

    public ListF<Delta> listDeltas(ListF<String> deltaIds) {
        return getDataFromBlackbox(new PassportDataSyncGetDeltas(deltaIds))
                .getDeltas();
    }


    // snapshots and records

    @Override
    public Option<Snapshot> getSnapshotO(RecordsFilter filter) {
        return getDatabaseO()
                .map(db -> new Snapshot(db, getRecords(db, filter, Option.of(db.rev))));
    }

    @Override
    public ListF<DataRecord> getDataRecords(RecordsFilter filter) {
        return getDataRecordsO(filter)
                .getOrElse(Cf.list());
    }

    @Override
    public Option<ListF<DataRecord>> getDataRecordsO(RecordsFilter filter) {
        return getDatabaseO().map(db -> getRecords(db, filter, Option.empty()));
    }

    private ListF<DataRecord> getRecords(Database database, RecordsFilter filter, Option<Long> dbRev) {
        return getDataFromBlackbox(PassportDataSyncGetObjects.fromRecordsFilter(databaseRef(), filter, dbRev), dbRev)
                .getDataRecords(database);
    }

    @Override
    public ListF<DataRecord> getDataRecordsByIds(SetF<DataRecordId> recordIds) {
        if (recordIds.isEmpty()) {
            return Cf.list();
        }

        return getDataFromBlackbox(PassportDataSyncGetObjects.fromRecordIds(recordIds))
                .getDataRecords(getDatabase());
    }

    @Override
    public int getDataRecordsCount(RecordsFilter filter) {
        return getDataRecordsO(filter)
                .map(List::size)
                .getOrElse(0);
    }

    @Override
    public void deleteDatabase(DatabaseDeletionMode deletionMode) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void deleteDatabase(DatabaseDeletionMode deletionMode, ChunkRateLimiter rateLimiter) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Database setDatabaseDescription(Option<String> newTitle) {
        return session.setDatabaseDescription(newTitle);
    }

    @Override
    public Database createDatabaseWithDescription(String title) {
        return session.createDatabaseWithDescription(title);
    }

    @Override
    public Database fixDatabaseRevision(long rev, long currentRev) {
        return session.fixDatabaseRevision(rev, currentRev);
    }

    @Override
    public void onDatabaseUpdate(long rev) {
        session.onDatabaseUpdate(rev);
    }

    @Override
    public void save(DatabaseChange change) {
        if (change.patchedDatabase().meta.recordsCount > BLACKBOX_DB_RECORD_LIMIT) {
            throw new DeltaValidationException(
                    "Can't store more than " + BLACKBOX_DB_RECORD_LIMIT + " for db " + databaseRef()
            );
        }
        try {
            save(new DiskAndPassportChanges(change));

        } catch (PassportDataSyncRevisionMismatchException e) {
            throw new StuckBehindDatabaseException(change.sourceDatabase(), e);
        }
    }

    private void save(DiskAndPassportChanges changes) {
        saveToDisk(changes.diskChange());
        saveToPassport(changes.passportChangeRequest());
    }

    private void saveToDisk(DatabaseChange change) {
        session.save(change);
    }

    public void saveRecordsToPassport(ListF<DataRecord> records, long newRevision, Option<Long> oldRevisionO) {
        saveToPassport(
                new PassportDataSyncChange(databaseRef(), newRevision)
                        .withRecordsToUpdate(records)
                        .withOldRevisionO(oldRevisionO)
        );
    }

    public void saveDeltasToPassport(Tuple2List<String, Delta> deltas, long newRevision, Option<Long> oldRevisionO) {
        saveToPassport(
                new PassportDataSyncChange(databaseRef(), newRevision)
                        .withDeltasToUpdate(deltas)
                        .withOldRevisionO(oldRevisionO)
        );
    }

    private void saveToPassport(PassportDataSyncChange changeRequest) {
        passportClient.save(passportUid.uid, changeRequest);
    }

    private class DiskAndPassportChanges {
        final DatabaseChange change;

        final Tuple2List<Delta, Delta> dataLessDeltaToFullDelta;

        DiskAndPassportChanges(DatabaseChange change) {
            this.change = change;
            this.dataLessDeltaToFullDelta =
                    change.getDeltas().zipWith(Delta::withoutDataAndWithRandomId)
                            .invert();
        }

        DatabaseChange diskChange() {
            return change.withoutRecords()
                    .withDeltas(dataLessDeltaToFullDelta.get1());
        }

        Tuple2List<String, Delta> idToDelta() {
            return dataLessDeltaToFullDelta.map1(Delta::getIdUnsafe);
        }

        PassportDataSyncChange passportChangeRequest() {
            return new PassportDataSyncChange(databaseRef(), change.patchedDatabase().rev)
                    .withRecordsToUpdate(change.getNewAndUpdatedRecords())
                    .withRecordsToDelete(change.getDeletedRecords())
                    .withDeltasToUpdate(idToDelta())
                    .withOldRevision(change.sourceDatabase().rev);
        }
    }
}
