package ru.yandex.chemodan.app.dataapi.core.datasources.passport;

import java.util.concurrent.TimeUnit;

import net.jodah.failsafe.FailsafeException;
import net.jodah.failsafe.RetryPolicy;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.StuckBehindDatabaseException;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaValidationException;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.PassportDataSyncClient;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.errors.PassportDataSyncRevisionMismatchException;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mockito.internal.verification.VerificationModeFactory.times;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class PassportSessionTest {
    private static final int RETRY_COUNT = 4;

    private static final long REQUEST_REV = 15L;

    private static final DataApiPassportUserId testUser = new DataApiPassportUserId(1L);

    private static final AppDatabaseRef testDbRef = new AppDatabaseRef("test", "test");

    private static final Database testDb = Database.consNew(testUser, testDbRef.consHandle("handle"))
            .withRev(REQUEST_REV);

    @Test
    public void throwDeltaValidationIfMoreThan1000Records() {
        Assert.assertThrows(() -> testSave(testDb.withRecordsCount(1001)), DeltaValidationException.class);
    }

    @Test
    public void save() {
        testSave(testDb);
    }

    @Test
    public void saveToStuckDatabase() {
        PassportDataSyncClient client = mock(PassportDataSyncClient.class);

        Mockito.doThrow(new PassportDataSyncRevisionMismatchException(Option.empty(), Option.empty()))
                .when(client).save(any(), any());

        Assert.assertThrows(() -> testSave(client, testDb),
                StuckBehindDatabaseException.class, e -> e.getDatabase().equals(testDb));
    }

    private void testSave(Database patchedDb) {
        testSave(mock(PassportDataSyncClient.class), patchedDb);
    }

    private void testSave(PassportDataSyncClient client, Database patchedDb) {
        DiskDataSource.Session diskSessionMock = consDiskSessionMock();
        PassportSession passportSession = new PassportSession(
                diskSessionMock,
                client,
                mock(UserMetaManager.class),
                new RetryPolicy()
        );
        DatabaseChange dbChange = mock(DatabaseChange.class);
        when(dbChange.sourceDatabase())
                .thenReturn(testDb);
        when(dbChange.patchedDatabase())
                .thenReturn(patchedDb);
        when(dbChange.getDeltas())
                .thenReturn(Cf.list());
        when(dbChange.withoutRecords())
                .thenReturn(dbChange);
        when(dbChange.getNewAndUpdatedRecords())
                .thenReturn(Cf.list());
        when(dbChange.getDeletedRecords())
                .thenReturn(Cf.list());

        passportSession.save(dbChange);

        verify(diskSessionMock, times(1))
                .save(Mockito.<DatabaseChange>any());
    }

    @Test
    public void retryIfPassportRevisionIsSmaller() {
        testGetDataRetries(REQUEST_REV - 1, RETRY_COUNT);
    }

    @Test
    public void retryIfPassportRevisionIsEmpty() {
        testGetDataRetries(Option.empty(), RETRY_COUNT);
    }

    @Test
    public void doNotRetryIfPassportRevisionIsEqual() {
        testGetDataRetries(REQUEST_REV, 0);
    }

    @Test
    public void doNotRetryIfPassportRevisionIsLarger() {
        testGetDataRetries(REQUEST_REV + 1, 0);
    }

    private void testGetDataRetries(long passportRev, int expectedRetries) {
        testGetDataRetries(Option.of(passportRev), expectedRetries);
    }

    private void testGetDataRetries(Option<Long> passportRev, int expectedRetries) {
        DiskDataSource.Session diskSessionMock = consDiskSessionMock();

        PassportDataSyncClient clientMock = mock(PassportDataSyncClient.class);
        when(clientMock.getData(eq(testUser), any()))
                .thenThrow(new PassportDataSyncRevisionMismatchException(Option.empty(), passportRev));

        PassportSession passportSession = new PassportSession(
                diskSessionMock,
                clientMock,
                mock(UserMetaManager.class),
                new RetryPolicy()
                        .withMaxRetries(RETRY_COUNT)
                        .withDelay(1, TimeUnit.MILLISECONDS)
        );

        try {
            passportSession.getSnapshotO(RecordsFilter.DEFAULT);
        } catch(FailsafeException ex) {
            // expected
        }

        verify(clientMock, times(1 + expectedRetries))
                .getData(eq(testUser), any());
    }

    private static DiskDataSource.Session consDiskSessionMock() {
        DiskDataSource.Session sessionMock = mock(DiskDataSource.Session.class);
        when(sessionMock.uid())
                .thenReturn(testUser);
        when(sessionMock.databaseRef())
                .thenReturn(testDbRef);
        when(sessionMock.databaseSpec())
                .thenReturn(new UserDatabaseSpec(testUser, testDbRef));
        when(sessionMock.getDatabaseO())
                .thenReturn(Option.of(testDb));
        return sessionMock;
    }
}
