package ru.yandex.chemodan.app.dataapi.core.datasources.passport.client;

import java.util.function.Consumer;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordIdSource;
import ru.yandex.chemodan.app.dataapi.api.data.record.VersionedDataApiRecord;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderFlatten;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 * <a href="https://wiki.yandex-team.ru/passport/api/bundle/datasync/">Запись DataSync-данных</a>
 */
@Bendable
public class PassportDataSyncChange extends DefaultObject {
    @BenderPart(name = "app_id", strictName = true)
    final String appName;

    @BenderPart(name = "database_id", strictName = true)
    final String databaseId;

    @BenderPart(name = "new_revision", strictName = true)
    final String newRevision;

    @BenderPart(name = "old_revision", strictName = true)
    final Option<String> oldRevisionStr;

    @BenderPart(name = "records_to_update", strictName = true)
    final ListF<PassportRecord> recordsToUpdate;

    @BenderPart(name = "records_to_delete", strictName = true)
    final ListF<PassportRecordId> recordsToDelete;

    @BenderPart(name = "deltas_to_update", strictName = true)
    final ListF<DeltaWithId> deltasToUpdate;

    @BenderPart(name = "deltas_to_delete", strictName = true)
    final ListF<String> deltasToDelete;

    public PassportDataSyncChange(DatabaseRef databaseRef, long newRevision) {
        this(databaseRef.dbAppId(), databaseRef.databaseId(), newRevision);
    }

    public PassportDataSyncChange(String appName, String databaseId, long newRevision) {
        this(appName, databaseId, String.valueOf(newRevision),
                Option.empty(), Cf.list(), Cf.list(), Cf.list(), Cf.list());
    }

    private PassportDataSyncChange(String appName, String databaseId,
            String newRevision, Option<String> oldRevisionStr,
            ListF<PassportRecord> recordsToUpdate, ListF<PassportRecordId> recordsToDelete,
            ListF<DeltaWithId> deltasToUpdate, ListF<String> deltasToDelete)
    {
        this.appName = appName;
        this.databaseId = databaseId;
        this.newRevision = newRevision;
        this.oldRevisionStr = oldRevisionStr;
        this.recordsToUpdate = recordsToUpdate;
        this.recordsToDelete = recordsToDelete;
        this.deltasToUpdate = deltasToUpdate;
        this.deltasToDelete = deltasToDelete;
    }

    public PassportDataSyncChange withOldRevision(long oldRevision) {
        return withOldRevisionO(Option.of(oldRevision));
    }

    public PassportDataSyncChange withOldRevisionO(Option<Long> oldRevisionO) {
        return build(b -> b.oldRevision = oldRevisionO.map(String::valueOf));
    }

    public PassportDataSyncChange withRecordsToUpdate(VersionedDataApiRecord... recordsToUpdate) {
        return withRecordsToUpdate(Cf.list(recordsToUpdate));
    }

    public PassportDataSyncChange withRecordsToUpdate(CollectionF<? extends VersionedDataApiRecord> recordsToUpdate)
    {
        return build(b -> b.recordsToUpdate = recordsToUpdate.map(PassportRecord::new));
    }

    public PassportDataSyncChange withRecordsToDelete(RecordIdSource... recordsToDelete) {
        return withRecordsToDelete(Cf.list(recordsToDelete));
    }

    public PassportDataSyncChange withRecordsToDelete(CollectionF<? extends RecordIdSource> recordsToDelete) {
        return build(b -> b.recordsToDelete = b.recordsToDelete.plus(recordsToDelete.map(PassportRecordId::new)));
    }

    public PassportDataSyncChange withDeltaToUpdate(String id, Delta delta) {
        return withDeltasToUpdate(Cf.list(new DeltaWithId(id, delta)));
    }

    public PassportDataSyncChange withDeltasToUpdate(Tuple2List<String, Delta> idToDelta) {
        return withDeltasToUpdate(idToDelta.map(DeltaWithId::fromIdWithDelta));
    }

    public PassportDataSyncChange withDeltasToUpdate(ListF<DeltaWithId> deltasToUpdate) {
        return build(b -> b.deltasToUpdate = deltasToUpdate);
    }

    public PassportDataSyncChange withDeltasToDelete(String... deltasToDelete) {
        return withDeltasToDelete(Cf.list(deltasToDelete));
    }

    public PassportDataSyncChange withDeltasToDelete(ListF<String> deltasToDelete) {
        return build(b -> b.deltasToDelete = deltasToDelete);
    }

    public Option<Long> oldRevision() {
        return oldRevisionStr.map(Long::parseLong);
    }

    private PassportDataSyncChange build(Consumer<Builder> changeF) {
        Builder builder = new Builder();
        changeF.accept(builder);
        return builder.build();
    }

    @Bendable
    public static class PassportRecord extends DefaultObject {
        @BenderFlatten
        final PassportRecordId id;

        @BenderPart(name = "record_value", strictName = true)
        final PassportDataSyncRecordValue value;

        PassportRecord(VersionedDataApiRecord record) {
            this(new PassportRecordId(record.id()), PassportDataSyncRecordValue.fromVersionedRecord(record));
        }

        PassportRecord(PassportRecordId id, PassportDataSyncRecordValue value) {
            this.id = id;
            this.value = value;
        }
    }

    @Bendable
    static class PassportRecordId extends DefaultObject {
        @BenderPart(name = "collection_id", strictName = true)
        final String collectionId;

        @BenderPart(name = "record_id", strictName = true)
        final String recordId;

        PassportRecordId(RecordIdSource recordIdSource) {
            this(recordIdSource.id());
        }

        PassportRecordId(RecordId recordId) {
            this(recordId.collectionId(), recordId.recordId());
        }

        PassportRecordId(String collectionId, String recordId) {
            this.collectionId = collectionId;
            this.recordId = recordId;
        }
    }

    @Bendable
    static class DeltaWithId {
        @BenderPart(name = "delta_id", strictName = true)
        final String id;

        @BenderPart(name = "delta_value", strictName = true)
        final Delta value;

        DeltaWithId(String id, Delta value) {
            this.id = id;
            this.value = value;
        }

        public static DeltaWithId fromIdWithDelta(Tuple2<String, Delta> stringDeltaTuple2) {
            return new DeltaWithId(stringDeltaTuple2.get1(), stringDeltaTuple2.get2());
        }
    }

    private class Builder {
        @BenderPart(name = "app_id")
        String appName = PassportDataSyncChange.this.appName;

        @BenderPart(name = "database_id")
        String databaseId = PassportDataSyncChange.this.databaseId;

        @BenderPart(name = "new_revision")
        String newRevision = PassportDataSyncChange.this.newRevision;

        @BenderPart(name = "old_revision")
        Option<String> oldRevision = PassportDataSyncChange.this.oldRevisionStr;

        @BenderPart(name = "records_to_update")
        ListF<PassportRecord> recordsToUpdate = PassportDataSyncChange.this.recordsToUpdate;

        @BenderPart(name = "records_to_delete")
        ListF<PassportRecordId> recordsToDelete = PassportDataSyncChange.this.recordsToDelete;

        @BenderPart(name = "deltas_to_update")
        ListF<DeltaWithId> deltasToUpdate = PassportDataSyncChange.this.deltasToUpdate;

        @BenderPart(name = "deltas_to_delete")
        ListF<String> deltasToDelete = PassportDataSyncChange.this.deltasToDelete;

        PassportDataSyncChange build() {
            return new PassportDataSyncChange(appName, databaseId, newRevision, oldRevision,
                    recordsToUpdate, recordsToDelete, deltasToUpdate, deltasToDelete);
        }
    }
}
