package ru.yandex.chemodan.app.dataapi.core.datasources.passport.client;

import org.junit.Test;

import ru.yandex.chemodan.app.dataapi.DataApiBenderUtils;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.field.NamedDataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleDataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class PassportDataSyncChangeSerializerTest {
    private static final BenderMapper basicMapper = DataApiBenderUtils.mapper();

    private static final DatabaseHandle DB_HANDLE =
            new AppDatabaseRef("passport", "passport")
                    .consHandle("handle1");

    private static final DataRecord RECORD = new SimpleDataRecord(
            new SimpleRecordId("test_collection", "test_record"),
            new DataFields(
                    new NamedDataField("integer_field", DataField.integer(123)),
                    new NamedDataField("boolean_field", DataField.bool(true))
            )
    ).toDataRecord(new DataApiPassportUserId(1), DB_HANDLE, 3);

    @Test
    public void basicTest() {
        DataRecordId recordId = new DataRecordId(
                DB_HANDLE, "collection1", "record1");
        Delta delta = Delta.empty()
                .plusDeleted(recordId);
        String recordValue = "{"
                + "\"rev\":3,"
                + "\"data\":{\"integer_field\":{\"type\":\"integer\",\"value\":123},\"boolean_field\":true}"
                + "}";
        String deltaValue = new String(basicMapper.serializeJson(delta), CharsetUtils.UTF8_CHARSET);
        String expectedJson = "{\"app_id\":\"test_app\",\"database_id\":\"test_db\""
                + ",\"new_revision\":\"7\",\"old_revision\":\"6\""
                + ",\"records_to_update\":[{"
                    + "\"collection_id\":\"test_collection\","
                    + "\"record_id\":\"test_record\","
                    + "\"record_value\":\"" + escapeDoubleQuotes(recordValue) + "\""
                + "}]"
                + ",\"records_to_delete\":["
                    + "{\"collection_id\":\"collection1\",\"record_id\":\"record1\"},"
                    + "{\"collection_id\":\"collection2\",\"record_id\":\"record2\"}"
                + "]"
                + ",\"deltas_to_update\":[{"
                    + "\"delta_id\":\"1234567890\","
                    + "\"delta_value\":\"" + escapeDoubleQuotes(deltaValue) + "\""
                + "}]"
                + ",\"deltas_to_delete\":[\"delta#1\",\"delta#2\"]" +
                "}";

        PassportDataSyncChange change = new PassportDataSyncChange("test_app", "test_db", 7)
                .withOldRevision(6)
                .withRecordsToUpdate(RECORD)
                .withRecordsToDelete(new SimpleRecordId("collection1", "record1"))
                .withRecordsToDelete(new SimpleRecordId("collection2", "record2"))
                .withDeltaToUpdate("1234567890", delta)
                .withDeltasToDelete("delta#1","delta#2")
                ;

        Assert.equals(expectedJson, serialize(change));
    }

    private static String escapeDoubleQuotes(String recordValue) {
        return recordValue.replaceAll("([\"])", "\\\\$1");
    }

    private String serialize(PassportDataSyncChange change) {
        return new String(PassportDataSyncChangeSerializer.serializeJson(change), CharsetUtils.UTF8_CHARSET);
    }
}
