package ru.yandex.chemodan.app.dataapi.core.datasources.passport.client;

import java.io.IOException;
import java.net.URI;

import org.apache.http.Header;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.errors.PassportDataSyncApiException;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.errors.PassportDataSyncBadResponseException;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.errors.PassportDataSyncBlackboxException;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.errors.PassportDataSyncResponseParseException;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.errors.PassportDataSyncRevisionMismatchException;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.errors.PassportDataSyncValueFormatException;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.request.PassportDataSyncGetRevision;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.request.PassportDataSyncRequest;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.request.PassportDataSyncRequestParam;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.parse.BenderParserException;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.io.http.Timeout;
import ru.yandex.misc.io.http.UriBuilder;
import ru.yandex.misc.io.http.apache.v4.ApacheHttpClientUtils;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.DefaultObject;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @see <a href="https://wiki.yandex-team.ru/passport/api/bundle/datasync/">Запись DataSync-данных</a>
 * @author Dmitriy Amelin (lemeh)
 */
public class PassportDataSyncClient {
    private static final Logger logger = LoggerFactory.getLogger(PassportDataSyncClient.class);
    private static final BenderMapper mapper = new BenderMapper();
    private static final String BLACKBOX_SERVICE = "Blackbox";
    private static final String PASSPORT_API_SERVICE = "Passport API";
    private final URI readUrl;
    private final URI saveUrl;
    private final HttpClient httpClient;
    private final ListF<Header> blackboxHeaders;

    private PassportDataSyncClient(Builder builder) {
        this.readUrl = builder.readUrl;
        this.saveUrl = builder.saveUrl;
        this.httpClient = builder.httpClient;
        this.blackboxHeaders = builder.blackboxHeaders;
    }

    /**
     * @see <a href="https://wiki.yandex-team.ru/passport/blackbox/#21.06.2016datasyncgetobjects">datasyncgetobjects</a>
     * @see <a href="https://wiki.yandex-team.ru/passport/blackbox/#21.06.2016datasyncgetdeltas>datasyncgetdeltas</a>
     */
    public PassportDataSyncData getData(PassportDataSyncRequest request) {
        try {
            HttpResponse response = httpClient.execute(consGetDataRequest(request));
            checkStatusCode(response, BLACKBOX_SERVICE);

            String responseBody = EntityUtils.toString(response.getEntity());

            Option<RuntimeException> errorResponse = tryParseBlackboxError(responseBody);
            if (errorResponse.isPresent()) {
                throw errorResponse.get();
            }

            PassportUid uid = request.getPassportUid();
            try {
                return PassportDataSyncParser.parse(responseBody)
                        .getOrThrow(uid)
                        .withDeltasSortedAs(request.getDeltaIds());
            } catch (BenderParserException ex) {
                throw new PassportDataSyncValueFormatException(
                        PassportDataSyncParser.parseWithStubbedValue(responseBody)
                                .getOrThrow(uid),
                        ex
                );
            }
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    public PassportDataSyncData getData(DataApiPassportUserId userId, PassportDataSyncRequestParam param) {
        return getData(new PassportDataSyncRequest(userId, param));
    }

    private HttpGet consGetDataRequest(PassportDataSyncRequest request) {
        HttpGet result = new HttpGet(consGetDataUrl(request));
        blackboxHeaders.forEach(result::addHeader);
        return result;
    }

    private URI consGetDataUrl(PassportDataSyncRequest request) {
        UriBuilder builder = new UriBuilder(readUrl)
                .addParam("uid", request.getPassportUid().toString());
        for(PassportDataSyncRequestParam param : request.getParams()) {
            builder.addParam(param.name(), param.value());
        }
        return builder.build();
    }

    /**
     * @see <a href="https://wiki.yandex-team.ru/passport/api/bundle/datasync/">Запись DataSync-данных</a>
     */
    public void save(PassportUid uid, PassportDataSyncChange change) {
        try {
            HttpResponse response = httpClient.execute(consSaveRequest(uid, change));
            checkStatusCode(response, PASSPORT_API_SERVICE);

            SaveResult result = parseSaveResult(response);
            if (!result.isOk()) {
                throw result.toException(change.oldRevision());
            }
        } catch (IOException e) {
            throw ExceptionUtils.translate(e);
        }
    }

    private static void checkStatusCode(HttpResponse response, String service) throws IOException {
        int statusCode = response.getStatusLine().getStatusCode();
        if (!HttpStatus.is2xx(statusCode)) {
            EntityUtils.consume(response.getEntity());
            throw new PassportDataSyncBadResponseException(service, statusCode);
        }
    }

    private HttpPost consSaveRequest(PassportUid uid, PassportDataSyncChange change) {
        ListF<NameValuePair> params = Cf.<NameValuePair>list()
                .plus(new BasicNameValuePair("uid", uid.toString()))
                .plus(new BasicNameValuePair("json_data", PassportDataSyncChangeSerializer.serializeJsonStr(change)));
        HttpPost request = new HttpPost(saveUrl);
        request.setEntity(new UrlEncodedFormEntity(params, CharsetUtils.UTF8_CHARSET));
        return request;
    }

    private static SaveResult parseSaveResult(HttpResponse response) throws IOException {
        try {
            String responseBody = EntityUtils.toString(response.getEntity());
            return mapper.parseJson(SaveResult.class, responseBody);
        } catch (RuntimeException e) {
            logger.warn("Could not parse passport response", e);
            throw new PassportDataSyncResponseParseException(e);
        }
    }

    private static Option<RuntimeException> tryParseBlackboxError(String responseBody) {
        try {
            return Option.of(mapper.parseJson(PassportDataSyncBlackboxException.class, responseBody))
                    .map(PassportDataSyncBlackboxException::getCauseOrSelf);
        } catch (RuntimeException e) {
            // Expected if no error occurred.
            // Blackbox API sucks: in case of error returns 200 OK and response in format other than normal response.
            return Option.empty();
        }
    }

    public Option<Long> getCurrentRevO(DataApiPassportUserId userId, DatabaseRef dbRef) {
        return getData(userId, new PassportDataSyncGetRevision(dbRef))
                .revisionO;
    }

    @Bendable
    private static class SaveResult extends DefaultObject {
        @BenderPart(name="status")
        final String status;

        @BenderPart(name="errors")
        final ListF<String> errors;

        SaveResult(String status, ListF<String> errors) {
            this.status = status;
            this.errors = errors;
        }

        boolean isOk() {
            return "ok".equals(status);
        }

        RuntimeException toException(Option<Long> requestRev) {
            if (errors.containsTs("revision.mismatch")) {
                return new PassportDataSyncRevisionMismatchException(requestRev, Option.empty());
            } else {
                return new PassportDataSyncApiException(status, errors);
            }
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder {
        private URI readUrl;
        private URI saveUrl;
        private HttpClient httpClient;
        private int maxConnectionCount = 32;
        private int soTimeoutMilliseconds = 5000;
        private int connectTimeoutMilliseconds = 1000;
        private ListF<Header> blackboxHeaders = Cf.arrayList();

        private Builder() {
        }

        public PassportDataSyncClient build() {
            Validate.notNull(readUrl, "readUrl can't be null");
            Validate.notNull(saveUrl, "saveUrl can't be null");
            httpClient = ApacheHttpClientUtils.multiThreadedClient(
                    new Timeout(soTimeoutMilliseconds, connectTimeoutMilliseconds),
                    maxConnectionCount);
            return new PassportDataSyncClient(this);
        }

        public Builder readUrl(URI readUrl) {
            this.readUrl = readUrl;
            return this;
        }

        public Builder readUrl(String blackboxUrl) {
            this.readUrl = new UriBuilder(blackboxUrl)
                    .addParam("method", "userinfo")
                    .addParam("userip", "127.0.0.1")
                    .addParam("format", "json")
                    .build();
            return this;
        }

        public Builder saveUrl(URI saveUrl) {
            this.saveUrl = saveUrl;
            return this;
        }

        public Builder saveUrl(String passportApiUrl, String passportApiConsumer) {
            this.saveUrl = new UriBuilder(passportApiUrl)
                    .appendPath("/1/bundle/datasync/save/")
                    .addParam("consumer", passportApiConsumer)
                    .build();
            return this;
        }

        public Builder blackboxHeader(Header blackboxHeader) {
            this.blackboxHeaders.add(blackboxHeader);
            return this;
        }

        public Builder maxConnectionCount(int threadCount) {
            this.maxConnectionCount = threadCount;
            return this;
        }

        public Builder soTimeoutMilliseconds(int soTimeoutMilliseconds) {
            this.soTimeoutMilliseconds = soTimeoutMilliseconds;
            return this;
        }

        public Builder connectTimeoutMilliseconds(int connectTimeoutMilliseconds) {
            this.connectTimeoutMilliseconds = connectTimeoutMilliseconds;
            return this;
        }
    }
}
