package ru.yandex.chemodan.app.dataapi.core.datasources.passport.client;

import org.apache.http.message.BasicHeader;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestRule;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.field.NamedDataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.VersionedDataApiRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.VersionedDataApiRecordImpl;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.GlobalDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.request.PassportDataSyncGetDeltas;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.request.PassportDataSyncGetObjects;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.request.PassportDataSyncRequest;
import ru.yandex.chemodan.util.test.SkipInYaTestRule;
import ru.yandex.chemodan.util.test.TestAppsAndUsers;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class PassportDataSyncClientTest {
    @Rule
    public final TestRule customRule = new SkipInYaTestRule();

    private static final String BLACKBOX_URL = "http://pass-test.yandex.ru/blackbox";
    private static final String PASSPORT_API_URL = "http://passport-test-internal.yandex.ru/";
    private static final String PASSPORT_API_CONSUMER = "disk-dataapi";

    private static final PassportUid TEST_UID = TestAppsAndUsers.testingUser2.uid;

    private final PassportDataSyncClient client = PassportDataSyncClient.builder()
            .readUrl(BLACKBOX_URL)
            .saveUrl(PASSPORT_API_URL, PASSPORT_API_CONSUMER)
            .blackboxHeader(new BasicHeader("Host", "pass-test.yandex.ru"))
            .build();

    @Test
    public void saveAndReadForGlobalDb() {
        testSaveAndRead(new GlobalDatabaseRef("passport"));
    }

    @Test
    public void saveAndReadForAppDb() {
        testSaveAndRead(new AppDatabaseRef("passport", "passport"));
    }

    public void testSaveAndRead(DatabaseRef databaseRef) {
        Option<Long> currentRevisionO = getCurrentRevisionO(databaseRef);
        long nextRev = currentRevisionO.map(r -> r + 1).getOrElse(1L);

        VersionedDataApiRecord record = new VersionedDataApiRecordImpl(
                "collectionId", "recordId" + System.currentTimeMillis(), 4,
                new DataFields(NamedDataField.string("field", "value"))
        );
        String deltaId = "delta" + System.currentTimeMillis();

        PassportDataSyncChange change = new PassportDataSyncChange(databaseRef, nextRev)
                .withOldRevisionO(currentRevisionO)
                .withRecordsToUpdate(record)
                .withDeltasToUpdate(Tuple2List.fromPairs(
                        deltaId, record.toInsertChange().toDelta()
                ));
        client.save(TEST_UID, change);

        DatabaseHandle dbHandle = databaseRef.consHandle("handle");
        PassportDataSyncData passportData = client.getData(
                new PassportDataSyncRequest(
                        new DataApiPassportUserId(TEST_UID),
                        PassportDataSyncGetObjects.fromRecordIds(
                                Cf.list(record.id().toDataRecordId(dbHandle)),
                                nextRev
                        ),
                        new PassportDataSyncGetDeltas(deltaId)
                )
        );

        ListF<VersionedDataApiRecord> actualRecords = passportData.getRecords();
        ListF<Delta> deltas = passportData.getDeltas();

        Assert.assertHasSize(1, actualRecords);
        Assert.equals(record, actualRecords.first());

        Assert.assertHasSize(1, deltas);
        Assert.equals(deltas.first(), record.toInsertChange().toDelta());
    }

    private Option<Long> getCurrentRevisionO(DatabaseRef databaseRef) {
        return client.getCurrentRevO(new DataApiPassportUserId(TEST_UID), databaseRef);
    }
}
