package ru.yandex.chemodan.app.dataapi.core.datasources.passport.client;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.DataApiBenderUtils;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.record.VersionedDataApiRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.VersionedDataApiRecordImpl;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.bender.EscapedJsonWrapperUnmarshaller;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.bender.StubValueUnmarshaller;
import ru.yandex.chemodan.util.bender.JsonFieldLevelUnmarshaller;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.config.BenderConfiguration;
import ru.yandex.misc.bender.config.BenderSettings;
import ru.yandex.misc.bender.config.CustomMarshallerUnmarshallerFactoryBuilder;
import ru.yandex.misc.bender.parse.BenderJsonNode;
import ru.yandex.misc.bender.parse.ParseResult;
import ru.yandex.misc.bender.parse.Unmarshaller;
import ru.yandex.misc.bender.parse.UnmarshallerContext;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class PassportDataSyncParser {
    private static final BenderMapper basicMapper = DataApiBenderUtils.mapper();

    private static final PassportDataSyncRecordValue stubRecordValue =
            new PassportDataSyncRecordValue(-1, DataFields.EMPTY);

    private static final BenderMapper stubValueMapper = consMapper(
            new StubValueUnmarshaller(stubRecordValue),
            new StubValueUnmarshaller(Delta.empty())
    );

    private static final BenderMapper mapper = consMapper(
            EscapedJsonWrapperUnmarshaller.fromParser(PassportDataSyncRecordValue.parser),
            new DeltaUnmarshaller()
    );

    public static PassportUsersDataSyncData parse(String value) {
        return parse(value, mapper);
    }

    public static PassportUsersDataSyncData parseWithStubbedValue(String value) {
        return parse(value, stubValueMapper);
    }

    private static PassportUsersDataSyncData parse(String value, BenderMapper mapper) {
        return new PassportUsersDataSyncData(mapper.parseJson(UserInfo.class, value)
                .users.filterMap(User::toUidToRecordsAndDeltas).toMap(t -> t));
    }

    @Bendable
    private static class UserInfo extends DefaultObject {
        @BenderPart(name = "users", strictName = true)
        final ListF<User> users;

        UserInfo(ListF<User> users) {
            this.users = users;
        }
    }

    @Bendable
    private static class User extends DefaultObject {
        @BenderPart(name = "uid", strictName = true)
        final UidInfo uid;

        @BenderPart(name = "datasync_objects", strictName = true)
        final ListF<DataSyncRecord> objects;

        @BenderPart(name = "datasync_deltas", strictName = true)
        final ListF<DataSyncDelta> deltas;

        @BenderPart(name = "datasync_revision", strictName = true)
        final Option<String> revisionO;

        User(UidInfo uid, ListF<DataSyncRecord> objects, ListF<DataSyncDelta> deltas, Option<String> revisionO) {
            this.uid = uid;
            this.objects = objects;
            this.deltas = deltas;
            this.revisionO = revisionO;
        }

        Option<Tuple2<PassportUid, PassportDataSyncData>> toUidToRecordsAndDeltas() {
            return uid.uid.zipWith(this::toRecordsAndDeltas).singleO();
        }

        private PassportDataSyncData toRecordsAndDeltas(PassportUid uid) {
            return new PassportDataSyncData(
                    uid,
                    objects.toTuple2List(DataSyncRecord::toVersionedRecordAndDbRev),
                    deltas.toTuple2List(DataSyncDelta::toTuple2),
                    revisionO.map(Long::parseLong)
            );
        }
    }

    @Bendable
    private static class UidInfo extends DefaultObject {
        @BenderPart(name = "value", strictName = true)
        final Option<PassportUid> uid;

        UidInfo(PassportUid uid) {
            this.uid = Option.of(uid);
        }
    }

    @Bendable
    private static class DataSyncRecord extends DefaultObject {
        @BenderPart(name = "app", strictName = true)
        final String app;
        @BenderPart(name = "db_id", strictName = true)
        final String dbId;
        @BenderPart(name = "collection_id", strictName = true)
        final String collectionId;
        @BenderPart(name = "record_id", strictName = true)
        final String recordId;
        @BenderPart(name = "revision", strictName = true)
        final String rev;
        @BenderPart(name = "value", strictName = true)
        final PassportDataSyncRecordValue value;

        DataSyncRecord(String app, String dbId, String collectionId, String recordId, String rev,
                PassportDataSyncRecordValue value)
        {
            this.app = app;
            this.dbId = dbId;
            this.collectionId = collectionId;
            this.recordId = recordId;
            this.rev = rev;
            this.value = value;
        }

        Tuple2<VersionedDataApiRecord, Long> toVersionedRecordAndDbRev() {
            return new Tuple2<>(toVersionedRecord(), Long.parseLong(rev));
        }

        private VersionedDataApiRecord toVersionedRecord() {
            return new VersionedDataApiRecordImpl(collectionId, recordId, value.rev, value.data);
        }
    }

    @Bendable
    private static class DataSyncDelta extends DefaultObject {
        @BenderPart(name = "delta_id", strictName = true)
        final String id;

        @BenderPart(name = "value", strictName = true)
        final Delta delta;

        DataSyncDelta(String id, Delta delta) {
            this.id = id;
            this.delta = delta;
        }

        public Tuple2<String, Delta> toTuple2() {
            return new Tuple2<>(id, delta);
        }
    }

    private static class DeltaUnmarshaller implements JsonFieldLevelUnmarshaller {
        @Override
        public ParseResult<Object> parseJsonNode(BenderJsonNode node, UnmarshallerContext context) {
            return ParseResult.result(parseDelta(node));
        }

        Delta parseDelta(BenderJsonNode node) {
            return basicMapper.parseJson(Delta.class, node.getValueAsString());
        }
    }

    private static BenderMapper consMapper(Unmarshaller recordValueUnmarshaller, Unmarshaller deltaUnmarshaller) {
        return new BenderMapper(new BenderConfiguration(
                new BenderSettings(),
                CustomMarshallerUnmarshallerFactoryBuilder
                        .cons()
                        .add(PassportDataSyncRecordValue.class, recordValueUnmarshaller)
                        .add(Delta.class, deltaUnmarshaller)
                        .build()
        ));
    }
}
