package ru.yandex.chemodan.app.dataapi.core.datasources.passport.client;

import org.junit.Test;

import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.field.NamedDataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleDataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.VersionedDataApiRecordImpl;
import ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.errors.PassportUserNotFoundException;
import ru.yandex.inside.passport.PassportUid;
import ru.yandex.misc.test.Assert;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class PassportDataSyncParserTest {
    @Test
    public void parseRecords() {
        String passportResponse = buildPassportResponse(
                "\"datasync_objects\":[{" +
                        "\"app\":\"passport\"," +
                        "\"db_id\":\"passport\"," +
                        "\"collection_id\":\"collectionId\"," +
                        "\"record_id\":\"recordId\"," +
                        "\"value\":\"{\\\"rev\\\":3,\\\"data\\\":{\\\"field\\\":\\\"value\\\"}}\"," +
                        "\"revision\":\"1983\"" +
                "}]");
        PassportDataSyncData data = PassportDataSyncData.consRecords(
                new PassportUid(4003323001L),
                Tuple2List.fromPairs(
                        new VersionedDataApiRecordImpl(
                                "collectionId", "recordId", 3,
                                new DataFields(NamedDataField.string("field", "value"))
                        ),
                        1983L
                )
        );

        Assert.equals(data, PassportDataSyncParser.parse(passportResponse).getOrThrow(data.uid));
    }

    @Test
    public void parseDeltas() {
        String passportResponse = buildPassportResponse(
                "\"datasync_deltas\":[{" +
                        "\"delta_id\":\"delta1\"," +
                        "\"value\":\"{\\\"changes\\\":[{" +
                        "\\\"op\\\":\\\"insert\\\"," +
                        "\\\"collectionId\\\":\\\"collectionId\\\"," +
                        "\\\"recordId\\\":\\\"recordId\\\"," +
                        "\\\"data\\\":{\\\"field\\\":\\\"value\\\"}}]}\"" +
                "}]"
        );
        PassportDataSyncData data = PassportDataSyncData.consDeltas(
                new PassportUid(4003323001L),
                Tuple2List.fromPairs(
                        "delta1",
                        new SimpleDataRecord(
                                new SimpleRecordId("collectionId", "recordId"),
                                new DataFields(NamedDataField.string("field", "value"))
                        )
                                .toInsertChange()
                                .toDelta()
                )
        );

        Assert.equals(data, PassportDataSyncParser.parse(passportResponse).getOrThrow(data.uid));
    }

    @Test
    public void parseRevision() {
        String passportResponse = buildPassportResponse("\"datasync_revision\": \"800\"");
        PassportDataSyncData data = PassportDataSyncData.consRevision(new PassportUid(4003323001L), 800L);

        Assert.equals(data, PassportDataSyncParser.parse(passportResponse).getOrThrow(data.uid));
    }

    @Test
    public void userNotFound() {
        String response = "{\"users\":["
                + "{\"id\":\"32046951\",\"uid\":{},\"karma\":{\"value\":0},\"karma_status\":{\"value\":0}}]}";

        Assert.assertThrows(() -> PassportDataSyncParser.parse(response).getOrThrow(PassportUid.cons(32046951)),
                PassportUserNotFoundException.class);
    }

    public static String buildPassportResponse(String extraData) {
        return "{\"users\":[{" +
                "\"id\":\"4003323001\"," +
                "\"uid\":{\"value\":\"4003323001\",\"lite\":false,\"hosted\":false}," +
                "\"login\":\"dataapi\"," +
                "\"have_password\":true," +
                "\"have_hint\":true," +
                "\"karma\":{\"value\":0}," +
                "\"karma_status\":{\"value\":0}," +
                extraData +
                "}]}";
    }
}
