package ru.yandex.chemodan.app.dataapi.core.datasources.passport.client.request;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.misc.bender.Bender;
import ru.yandex.misc.bender.annotation.Bendable;
import ru.yandex.misc.bender.annotation.BenderPart;
import ru.yandex.misc.bender.serialize.BenderSerializer;
import ru.yandex.misc.lang.CharsetUtils;
import ru.yandex.misc.lang.DefaultObject;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class PassportDataSyncGetObjects implements PassportDataSyncRequestParam {
    private static final String EXPECTING_ALL_COND_MSG =
            "Could not build request to passport from records filter: %s condition MUST be ALL";

    private static final String EXPECTING_ALL_COND_OR_IDS_MSG = EXPECTING_ALL_COND_MSG + " or with concrete ids";

    private static final BenderSerializer<RecordFilter> serializer = Bender.serializer(RecordFilter.class);

    private final ListF<RecordFilter> filters;

    private PassportDataSyncGetObjects(RecordFilter... filters) {
        this(Cf.list(filters));
    }

    private PassportDataSyncGetObjects(ListF<RecordFilter> filters) {
        this.filters = filters;
    }

    public static PassportDataSyncGetObjects fromRecordsFilter(
            DatabaseRef dbRef,
            RecordsFilter filter,
            Option<Long> dbRevO)
    {
        if (!filter.getRecordCond().isAll()) {
            throw new IllegalArgumentException(String.format(EXPECTING_ALL_COND_MSG, "data"));
        }

        if (!filter.getRecordIdCond().isAllOrWithIds()) {
            throw new IllegalArgumentException(String.format(EXPECTING_ALL_COND_OR_IDS_MSG, "recordId"));
        }

        if (!filter.getCollectionIdCond().isAllOrWithIds()) {
            throw new IllegalArgumentException(String.format(EXPECTING_ALL_COND_OR_IDS_MSG, "collectionId"));
        }

        if (filter.getRecordIdCond().isWithIds() && !filter.getCollectionIdCond().isWithIds()) {
            throw new IllegalArgumentException(
                    "CollectionId condition MUST be with ids when recordId condition with ids");
        }

        if (filter.getCollectionIdCond().isAll() && filter.getRecordIdCond().isAll()) {
            return new PassportDataSyncGetObjects(new RecordFilter(dbRef));
        }

        ListF<CollectionRef> collectionRefs = filter.getCollectionIdCond().idsO.get()
                .map(dbRef::consColRef);

        if (filter.getRecordIdCond().isAll()) {
            return new PassportDataSyncGetObjects(collectionRefs.map(RecordFilter::new));
        }

        SetF<String> recordIds = filter.getRecordIdCond().idsO.get();
        ListF<RecordRef> recordRefs = collectionRefs
                .map(colRef -> recordIds.map(colRef::consRecordRef))
                .flatten();
        return new PassportDataSyncGetObjects(
                recordRefs.map(RecordFilter::new)
                        .map(f -> f.withRev(dbRevO))
        );
    }

    public static PassportDataSyncGetObjects fromRecordIds(CollectionF<DataRecordId> recordIds) {
        return new PassportDataSyncGetObjects(recordIds.map(RecordFilter::new));
    }

    public static PassportDataSyncGetObjects fromRecordIds(CollectionF<DataRecordId> recordIds, long dbRev) {
        return new PassportDataSyncGetObjects(
                recordIds.map(RecordFilter::new)
                        .map(f -> f.withRev(dbRev))
        );
    }

    @Override
    public PassportDataSyncRequestType type() {
        return PassportDataSyncRequestType.OBJECTS;
    }

    @Override
    public String value() {
        return new String(serializer.serializeListJson(filters), CharsetUtils.UTF8_CHARSET);
    }

    @Bendable
    private static class RecordFilter extends DefaultObject {
        @BenderPart(name = "app_id", strictName = true)
        final Option<String> app;

        @BenderPart(name = "db_id", strictName = true)
        final Option<String> databaseId;

        @BenderPart(name = "coll_id", strictName = true)
        final Option<String> collectionId;

        @BenderPart(name = "rec_id", strictName = true)
        final Option<String> recordId;

        @BenderPart(name = "rev", strictName = true)
        final Option<String> rev;

        RecordFilter(DatabaseRef databaseRef) {
            this(databaseRef, Option.empty(), Option.empty());
        }

        RecordFilter(CollectionRef collectionRef) {
            this(collectionRef.dbRef(), Option.of(collectionRef.collectionId()), Option.empty());
        }

        RecordFilter(RecordRef recordRef) {
            this(recordRef.dbRef(), Option.of(recordRef.collectionId()), Option.of(recordRef.recordId()));
        }

        RecordFilter(DataRecordId recordId) {
            this(recordId.handle.dbRef, Option.of(recordId.collectionId()), Option.of(recordId.recordId()));
        }

        RecordFilter(DatabaseRef databaseRef, Option<String> collectionId, Option<String> recordId) {
            this(Option.of(databaseRef.dbAppId()),
                    Option.of(databaseRef.databaseId()),
                    collectionId, recordId, Option.empty());
        }

        RecordFilter(Option<String> app, Option<String> databaseId,
                Option<String> collectionId, Option<String> recordId,
                Option<String> rev)
        {
            this.app = app;
            this.databaseId = databaseId;
            this.collectionId = collectionId;
            this.recordId = recordId;
            this.rev = rev;
        }

        RecordFilter withRev(long rev) {
            return withRev(Option.of(rev));
        }

        RecordFilter withRev(Option<Long> rev) {
            return new RecordFilter(app, databaseId, collectionId, recordId, rev.map(String::valueOf));
        }
    }
}
