package ru.yandex.chemodan.app.dataapi.core.datasources.yamoney;

import java.io.IOException;
import java.net.URI;

import org.apache.http.Header;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.message.BasicHeader;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.CollectionF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function2;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotPojo;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaValidationException;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltasAppliedDatabase;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.web.DeltasGoneException;
import ru.yandex.chemodan.app.dataapi.web.direct.a3.DirectDataApiBenderUtils;
import ru.yandex.chemodan.app.dataapi.web.pojo.DatabasePojo;
import ru.yandex.chemodan.util.exception.A3ExceptionWithStatus;
import ru.yandex.commune.a3.security.UnauthorizedException;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.io.InputStreamX;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class YaMoneyDataApiClient {
    private static final Logger logger = LoggerFactory.getLogger(YaMoneyDataApiClient.class);

    private static final BasicHeader JSON_CONTENT_TYPE_HEADER =
            new BasicHeader(HttpHeaders.CONTENT_TYPE, ContentType.APPLICATION_JSON.getMimeType());

    private static final int SC_429_TOO_MANY_REQUESTS = 429;

    private static final SetF<Integer> COMMON_STATUS_CODES = Cf.set(
            HttpStatus.SC_400_BAD_REQUEST,
            HttpStatus.SC_401_UNAUTHORIZED,
            HttpStatus.SC_403_FORBIDDEN,
            HttpStatus.SC_404_NOT_FOUND,
            HttpStatus.SC_406_NOT_ACCEPTABLE,
            SC_429_TOO_MANY_REQUESTS);

    private static final SetF<Integer> COMMON_DB_STATUS_CODES =
            COMMON_STATUS_CODES.plus1(HttpStatus.SC_423_LOCKED);

    private static final SetF<Integer> GET_OR_CREATE_DB_STATUS_CODES =
            COMMON_DB_STATUS_CODES.plus(HttpStatus.SC_507_INSUFFICIENT_STORAGE);

    private static final SetF<Integer> DELTA_APPLY_STATUS_CODES = COMMON_STATUS_CODES.plus(
            HttpStatus.SC_409_CONFLICT,
            HttpStatus.SC_410_GONE,
            HttpStatus.SC_412_PRECONDITION_FAILED,
            HttpStatus.SC_415_UNSUPPORTED_MEDIA_TYPE,
            HttpStatus.SC_423_LOCKED,
            HttpStatus.SC_507_INSUFFICIENT_STORAGE
    );

    private static final BenderMapper mapper = DirectDataApiBenderUtils.mapper();

    private final HttpClient httpClient;

    private final URI serviceUrl;

    public YaMoneyDataApiClient(URI serviceUrl, HttpClient httpClient) {
        this.serviceUrl = serviceUrl;
        this.httpClient = httpClient;
    }

    public Database getOrCreateDatabase(DataApiUserId uid, DatabaseRef dbRef) {
        return new Request(uid, dbRef, "PUT")
                .withPassThroughStatusCodes(GET_OR_CREATE_DB_STATUS_CODES)
                .executeAndParse((response, statusCode) -> parseDatabase(uid, dbRef, response, statusCode));
    }

    public Option<Database> getDatabase(DataApiUserId uid, DatabaseRef dbRef) {
        return new Request(uid, dbRef, "GET")
                .withPassThroughStatusCodes(COMMON_DB_STATUS_CODES)
                .withDoNotThrowStatusCodes(HttpStatus.SC_404_NOT_FOUND)
                .executeAndParse((response, statusCode) -> parseDatabaseO(uid, dbRef, response, statusCode));
    }

    public Option<SnapshotPojo> getSnapshotO(DataApiUserId uid, DatabaseRef dbRef) {
        return new Request(uid, dbRef, "GET", "snapshot")
                .withPassThroughStatusCodes(COMMON_STATUS_CODES)
                .withDoNotThrowStatusCodes(HttpStatus.SC_404_NOT_FOUND)
                .executeAndParse(this::parseSnapshotO);
    }

    public DeltasAppliedDatabase applyDeltas(DataApiUserId uid, DatabaseRef dbRef, Database srcDb, Delta delta) {
        try {
            return new Request(uid, dbRef, "POST", "deltas")
                    .withPassThroughStatusCodes(DELTA_APPLY_STATUS_CODES)
                    .withHeader(HttpHeaders.IF_MATCH, String.valueOf(srcDb.rev))
                    .withBody(mapper.serializeJson(delta))
                    .executeAndParse(this::parseDeltasAppliedDatabase);
        } catch(A3ExceptionWithStatus ex) {
            switch(ex.getHttpStatusCode()) {
                case HttpStatus.SC_410_GONE:
                    throw new DeltasGoneException();

                case HttpStatus.SC_400_BAD_REQUEST:
                case HttpStatus.SC_406_NOT_ACCEPTABLE:
                case HttpStatus.SC_423_LOCKED:
                    throw new DeltaValidationException(
                            "Got status code#" + ex.getHttpStatusCode() + " from Ya.Money");

                default:
                    throw ex;
            }
        }
    }

    public void deleteDatabase(DataApiUserId uid, DatabaseRef dbRef) {
        new Request(uid, dbRef, "DELETE")
                .execute();
    }

    private Option<Database> parseDatabaseO(DataApiUserId uid, DatabaseRef dbRef, byte[] data, Integer statusCode) {
        return statusCode == HttpStatus.SC_404_NOT_FOUND
                ? Option.empty()
                : Option.of(parseDatabase(uid, dbRef, data, statusCode));
    }

    Database parseDatabase(DataApiUserId uid, DatabaseRef dbRef, byte[] data, Integer statusCode) {
        DatabasePojo pojo = mapper.parseJson(DatabasePojo.class, data);
        return statusCode == HttpStatus.SC_201_CREATED
                ? pojo.toNewDatabase(uid, dbRef)
                : pojo.toExistingDatabase(uid, dbRef);
    }

    private Option<SnapshotPojo> parseSnapshotO(byte[] bytes, Integer statusCode) {
        return statusCode == HttpStatus.SC_404_NOT_FOUND
                ? Option.empty()
                : Option.of(mapper.parseJson(SnapshotPojo.class, bytes));
    }

    private DeltasAppliedDatabase parseDeltasAppliedDatabase(byte[] response, int statusCode) {
        return mapper.parseJson(DeltasAppliedDatabase.class, response);
    }

    private class Request {
        final DataApiPassportUserId uid;
        final DatabaseRef dbRef;
        final String method;
        final Option<String> resourceO;
        final Option<byte[]> inputO;
        final SetF<Integer> doNotThrowStatusCodes;
        final SetF<Integer> passThroughStatusCodes;
        final ListF<Header> extraHeaders;

        private Request(DataApiUserId uid, DatabaseRef dbRef, String method) {
            this(uid, dbRef, method, Option.empty());
        }

        private Request(DataApiUserId uid, DatabaseRef dbRef, String method, String resource) {
            this(uid, dbRef, method, Option.of(resource));
        }

        private Request(DataApiUserId uid, DatabaseRef dbRef, String method, Option<String> resourceO) {
            this(uid, dbRef, method, resourceO, Option.empty(), Cf.set(), Cf.set(), Cf.list());
        }

        private Request(DataApiUserId uid, DatabaseRef dbRef, String method, Option<String> resourceO,
                Option<byte[]> inputO, SetF<Integer> passThroughStatusCodes, SetF<Integer> doNotThrowStatusCodes,
                ListF<Header> extraHeaders)
        {
            if (!(uid instanceof DataApiPassportUserId)) {
                throw new UnauthorizedException("Expecting passport user");
            }

            DataApiPassportUserId passportUid = (DataApiPassportUserId) uid;

            if (!passportUid.oAuthToken.isPresent()) {
                throw new UnauthorizedException("OAuth token is not specified");
            }

            this.uid = passportUid;
            this.method = method;
            this.dbRef = dbRef;
            this.resourceO = resourceO;
            this.inputO = inputO;
            this.passThroughStatusCodes = passThroughStatusCodes;
            this.doNotThrowStatusCodes = doNotThrowStatusCodes;
            this.extraHeaders = extraHeaders;
        }

        Request withPassThroughStatusCodes(CollectionF<Integer> statusCodes) {
            return new Request(uid, dbRef, method, resourceO, inputO,
                    statusCodes.unique(), doNotThrowStatusCodes, extraHeaders);
        }

        Request withDoNotThrowStatusCodes(Integer... statusCodes) {
            return new Request(uid, dbRef, method, resourceO, inputO,
                    passThroughStatusCodes, Cf.set(statusCodes), extraHeaders);
        }

        Request withBody(byte[] input) {
            return new Request(uid, dbRef, method, resourceO, Option.of(input),
                    passThroughStatusCodes, doNotThrowStatusCodes, extraHeaders);
        }

        Request withHeader(String name, String value) {
            return withHeader(new BasicHeader(name, value));
        }

        Request withHeader(Header header) {
            return new Request(uid, dbRef, method, resourceO, inputO,
                    passThroughStatusCodes, doNotThrowStatusCodes, extraHeaders.plus1(header));
        }

        <T> T executeAndParse(Function2<byte[], Integer, T> mapperF) {
            try {
                return httpClient.execute(httpUriRequest(), response -> parseResponse(response, mapperF));
            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
        }

        void execute() {
            try {
                httpClient.execute(httpUriRequest(), response -> parseResponse(response, (bytes, integer) -> null));
            } catch (IOException e) {
                throw ExceptionUtils.translate(e);
            }
        }

        private <T> T parseResponse(HttpResponse resp, Function2<byte[], Integer, T> mapperF) throws IOException {
            validateStatusCode(resp);
            return mapperF.apply(readContent(resp), resp.getStatusLine().getStatusCode());
        }

        private void validateStatusCode(HttpResponse response) {
            int statusCode = response.getStatusLine().getStatusCode();
            if (HttpStatus.is2xx(statusCode) || doNotThrowStatusCodes.containsTs(statusCode)) {
                return;
            }

            String responseBody = readContentQuietly(response);
            if (passThroughStatusCodes.containsTs(statusCode)) {
                throw new A3ExceptionWithStatus("ya-money-error",
                        "Ya.Money responded with sc#" + statusCode, statusCode);
            } else {
                throw new IllegalStateException(String.format(
                        "Got unexpected response from Ya.Money backend: sc=%s, message=%s",
                        statusCode, responseBody
                ));
            }
        }

        private byte[] readContent(HttpResponse response) throws IOException {
            if (hasEmptyBody(response)) {
                return new byte[0];
            }

            return new InputStreamX(response.getEntity().getContent())
                    .readBytes();
        }

        private boolean hasEmptyBody(HttpResponse response) {
            return !Option.ofNullable(response.getEntity())
                    .isMatch(entity -> entity.getContentLength() != 0);
        }

        private String readContentQuietly(HttpResponse response) {
            try {
                return new String(readContent(response));
            } catch (Exception e) {
                logger.info("Error while reading error response from Ya.Money backend {}",
                        ExceptionUtils.prettyPrint(e));
                return "";
            }
        }

        HttpUriRequest httpUriRequest() {
            throw new YaMoneyRequestApiException("deprecated api");
        }
    }

}
