package ru.yandex.chemodan.app.dataapi.core.datasources.yamoney;

import org.apache.http.impl.client.HttpClients;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestRule;
import org.mockito.Mockito;

import ru.yandex.chemodan.app.dataapi.api.data.field.NamedDataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleDataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.SimpleRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.core.xiva.DataApiXivaPushSender;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.util.test.SkipInYaTestRule;
import ru.yandex.chemodan.util.test.TestAppsAndUsers;
import ru.yandex.misc.io.http.UrlUtils;
import ru.yandex.misc.property.PropertiesHolder;
import ru.yandex.misc.random.Random2;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class YaMoneyDatabaseSessionTest {
    @Rule
    public final TestRule customRule = new SkipInYaTestRule();

    static {
        DataApiTestSupport.initProperties();
    }

    private static final SimpleDataRecord SAMPLE_RECORD =
            new SimpleDataRecord("userdata", "1",
                    NamedDataField.integer("data_type", 1),
                    NamedDataField.string("data_value", randomNumeric(10)),
                    NamedDataField.bool("subscription", false)
            );

    private final YamoneyDatabaseDataSource dataSource = consDataSource();

    private final UserDatabaseSpec dbSpec = new UserDatabaseSpec(
            DataApiPassportUserId
                    .fromToken(TestAppsAndUsers.randomTestingUser().getTokenFor(TestAppsAndUsers.testingApp)),
            YaMoneyDatabaseRef.DATA.dbRef()
    );

    @Test(expected = YaMoneyRequestApiException.class)
    public void insertSingleRecord() {
        testApplyDelta(
                Delta.fromNewRecords(
                        SAMPLE_RECORD
                )
        );
    }

    @Test(expected = YaMoneyRequestApiException.class)
    public void deltaValidationWhenIncorrectCollection() {
        testApplyDelta(
                Delta.fromNewRecords(
                        SAMPLE_RECORD.withRecordId(new SimpleRecordId("wrong-collection-", "1"))
                )
        );
    }

    @Test(expected = YaMoneyRequestApiException.class)
    public void deltaValidationWhenIncorrectValue() {
        testApplyDelta(
                Delta.fromNewRecords(
                        SAMPLE_RECORD.withData(
                                SAMPLE_RECORD.data.plus(
                                        NamedDataField.string("data_value", randomNumeric(3))
                                )
                        )
                )
        );
    }

    @Test(expected = YaMoneyRequestApiException.class)
    public void deltaValidationWhenIncorrectType() {
        testApplyDelta(
                Delta.fromNewRecords(
                        SAMPLE_RECORD.withData(
                                SAMPLE_RECORD.data.plus(
                                        NamedDataField.integer("data_type", 4)
                                )
                        )
                )
        );
    }

    private void testApplyDelta(Delta delta) {
        YamoneyDatabaseSession session = dataSource.openSession(dbSpec);
        session.getOrCreateDatabase();
        session.save(
                session.getSnapshotO(RecordsFilter.DEFAULT)
                        .get()
                        .toPatchable()
                        .patch(delta)
                        .toDatabaseChange()
        );
    }

    private static YamoneyDatabaseDataSource consDataSource() {
        return new YamoneyDatabaseDataSource(
                new YaMoneyDataApiClient(
                        UrlUtils.uri(PropertiesHolder.properties().getProperty("yamoney.datasync-adapter.url")),
                        HttpClients.createDefault()
                ),
                Mockito.mock(DataApiXivaPushSender.class)
        );
    }

    private static String randomNumeric(int size) {
        return Random2.threadLocal()
                .nextString(size, "0123456789");
    }
}
