package ru.yandex.chemodan.app.dataapi.core.datasources.yamoney;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.function.Function0V;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldType;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataApiRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.SnapshotPojo;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceSession;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.core.datasources.disk.DiskDataSource;
import ru.yandex.chemodan.app.dataapi.core.datasources.yamoney.records.AbstractYaMoneyEntity;
import ru.yandex.chemodan.app.dataapi.core.datasources.yamoney.records.UserDataRecord;
import ru.yandex.chemodan.app.dataapi.core.xiva.DataApiXivaPushSender;
import ru.yandex.chemodan.app.dataapi.web.DeltasGoneException;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.commune.util.RetryUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Dmitriy Amelin (lemeh)
 */
public class YamoneyDatabaseSession implements DataSourceSession {
    private static final Logger logger = LoggerFactory.getLogger(DiskDataSource.class);

    private static final String TITLE_FIELD_NAME = "title";

    private final UserDatabaseSpec databaseSpec;

    private final YaMoneyDatabaseRef databaseRef;

    private final YaMoneyDataApiClient client;

    private final DataApiXivaPushSender xivaPushSender;

    YamoneyDatabaseSession(UserDatabaseSpec databaseSpec, YaMoneyDataApiClient client,
            DataApiXivaPushSender xivaPushSender)
    {
        this.databaseSpec = databaseSpec;
        this.databaseRef = YaMoneyDatabaseRef.byDatabaseRef(databaseSpec.databaseRef());
        this.client = client;
        this.xivaPushSender = xivaPushSender;
    }

    @Override
    public UserDatabaseSpec databaseSpec() {
        return databaseSpec;
    }

    @Override
    public Database getOrCreateDatabase() {
        return client.getOrCreateDatabase(uid(), databaseRef())
                .withHandle(generateHandle());
    }

    @Override
    public Database createDatabase() {
        Database database = getOrCreateDatabase();
        if (!database.isNew) {
            throw databaseRef().consExistsException();
        }
        return database;
    }

    @Override
    public Option<Database> getDatabaseO() {
        return client.getDatabase(uid(), databaseRef())
                .map(db -> db.withHandle(generateHandle()));
    }

    @Override
    public void save(DatabaseChange change) {
        client.applyDeltas(uid(), databaseRef(), change.sourceSnapshot.database, change.toSingleDelta());
        // deltas is not saved by intention
    }

    @Override
    public Option<Snapshot> getSnapshotO(RecordsFilter filter) {
        return client.getSnapshotO(uid(), databaseRef())
                .map(this::toDomainObject);
    }

    private Snapshot toDomainObject(SnapshotPojo pojo) {
        Snapshot snapshot = pojo.toDomainObject(uid(), databaseRef())
                .withHandle(generateHandle());
        return new Snapshot(snapshot.database, snapshot.records.mutate(this::removeTitleIfNull));
    }

    private DataRecord removeTitleIfNull(DataRecord dataRecord) {
        Option<DataField> titleO = dataRecord.getFields().getO(TITLE_FIELD_NAME);
        if (titleO.isPresent() && titleO.get().fieldType == DataFieldType.NULL) {
            return dataRecord.withData(
                    dataRecord.getFields().without(TITLE_FIELD_NAME)
            );
        }
        return dataRecord;
    }

    @Override
    public Option<ListF<DataRecord>> getDataRecordsO(RecordsFilter filter) {
        return getSnapshotO(filter)
                .map(snapshot -> snapshot.records().toList());
    }

    private DatabaseHandle generateHandle() {
        return databaseRef.generateHandle(uid());
    }

    @Override
    public int getDataRecordsCount(RecordsFilter filter) {
        return getSnapshotO(filter)
                .map(Snapshot::recordCount)
                .getOrElse(0);
    }

    @Override
    public ListF<DataRecord> getDataRecordsByIds(SetF<DataRecordId> recordIds) {
        return getDataRecords(RecordsFilter.DEFAULT);
    }

    @Override
    public void onDatabaseUpdate(long rev) {
        xivaPushSender.doSendPush(uid(), databaseRef.dbRef(), rev);
        logger.info("Received and pushed to Xiva notification that user={} database={} was updated to revision#{}",
                uid(), databaseRef(), rev);
    }


    /**
     * Unsupported operations
     */

    @Override
    public Delta getDelta(long rev) {
        throw NotFoundException.consDeltaNotFound(rev);
    }

    @Override
    public ListF<Delta> listDeltas(long fromRev, int limit) {
        throw new DeltasGoneException();
    }

    @Override
    public void deleteDatabase(DatabaseDeletionMode deletionMode) {
        throw new UnsupportedOperationException();
    }

    @Override
    public void deleteDatabase(DatabaseDeletionMode deletionMode, ChunkRateLimiter rateLimiter) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Database setDatabaseDescription(Option<String> newTitle) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Database createDatabaseWithDescription(String title) {
        throw new UnsupportedOperationException();
    }

    @Override
    public Database fixDatabaseRevision(long rev, long currentRev) {
        throw new UnsupportedOperationException();
    }

    public void cleanUpAfterTests() {
        if (!getDatabaseO().isPresent()) {
            return;
        }

        switch(databaseRef) {
            case DATA:
                removeSubscriptionWithRetries();

            default:
                client.deleteDatabase(uid(), databaseRef.dbRef());
        }
    }

    private void removeSubscriptionWithRetries() {
        RetryUtils.retryOrThrow(logger, 5,
                Function0V.wrap(this::removeSubscription)
                        .asFunction0ReturnNull(),
                ex -> !(ex instanceof DeltasGoneException)
        );
    }

    private void removeSubscription() {
        Snapshot snapshot = getSnapshotO(RecordsFilter.DEFAULT).get();
        if (snapshot.isEmpty()) {
            return;
        }

        ListF<RecordChange> records = snapshot
                .records
                .iterator()
                .map(UserDataRecord::parse)
                .map(UserDataRecord::withoutSubscription)
                .map(AbstractYaMoneyEntity::toSimpleRecord)
                .map(DataApiRecord::toSetChange)
                .toList();
        client.applyDeltas(uid(), databaseRef.dbRef(), snapshot.database, new Delta(records));
    }
}
