package ru.yandex.chemodan.app.dataapi.core.datasources.yamoney.records;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFields;
import ru.yandex.chemodan.app.dataapi.api.data.field.NamedDataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.core.datasources.yamoney.YaMoneyCollectionRef;
import ru.yandex.misc.enums.IntEnum;
import ru.yandex.misc.enums.IntEnumResolver;

/**
 * Используется в тестах.
 *
 * @author Dmitriy Amelin (lemeh)
 */
public class UserDataRecord extends AbstractYaMoneyEntity {
    public final String recordId;

    final DataType type;

    final String value; // doc number: "^[\\p{Letter}0-9]{10}$"

    final Option<String> title;

    final boolean subscription;

    private UserDataRecord(String recordId, DataType type, String value, Option<String> title, boolean subscription) {
        this.recordId = recordId;
        this.type = type;
        this.value = value;
        this.title = title;
        this.subscription = subscription;
    }

    private UserDataRecord(String recordId, DataType type, String value) {
        this(recordId, type, value, Option.empty(), false);
    }

    public static UserDataRecord parse(DataRecord record) {
        DataFields fields = record.getFields();
        int data_type = fields.get("data_type").integerValue().intValue();
        return new UserDataRecord(
                record.getRecordId(),
                DataType.parse(data_type),
                fields.get("data_value").stringValue(),
                fields.getO("title").map(DataField::stringValue),
                fields.get("subscription").booleanValue()
        );
    }

    public static UserDataRecord drivingLicense(String recordId, String value) {
        return new UserDataRecord(recordId, DataType.DRIVING_LICENSE, value);
    }

    public static UserDataRecord registrationCertificate(String recordId, String value) {
        return new UserDataRecord(recordId, DataType.VEHICLE_REGISTRATION_CERTIFICATE, value);
    }

    public static UserDataRecord registrationNumber(String recordId, String value) {
        return new UserDataRecord(recordId, DataType.VEHICLE_REGISTRATION_NUMBER, value);
    }

    public UserDataRecord withTitle(String title) {
        return new UserDataRecord(recordId, type, value, Option.of(title), subscription);
    }

    public UserDataRecord withSubscription() {
        if (type == DataType.VEHICLE_REGISTRATION_NUMBER) {
            // https://st.yandex-team.ru/CHEMODAN-30713#1468304516000
            throw new IllegalStateException("Subscription to vehicle number is not allowed");
        }

        return new UserDataRecord(recordId, type, value, title, true);
    }

    public UserDataRecord withoutSubscription() {
        return new UserDataRecord(recordId, type, value, title, false);
    }

    @Override
    public YaMoneyCollectionRef collectionRef() {
        return YaMoneyCollectionRef.DATA;
    }

    @Override
    protected String recordId() {
        return recordId;
    }

    @Override
    protected DataFields data() {
        return new DataFields(
                NamedDataField.integer("data_type", type.value),
                NamedDataField.string("data_value", value),
                NamedDataField.bool("subscription", subscription)
        ).plusO(
                title.map(t -> NamedDataField.string("title", t))
        );
    }

    enum DataType implements IntEnum {
        DRIVING_LICENSE(1),
        VEHICLE_REGISTRATION_CERTIFICATE(2),
        VEHICLE_REGISTRATION_NUMBER(3);

        final int value;

        DataType(int value) {
            this.value = value;
        }

        @Override
        public int value() {
            return value;
        }

        public static DataType parse(int value) {
            return R.fromValue(value);
        }

        private static final IntEnumResolver<DataType> R = IntEnumResolver.r(DataType.class);
    }
}
