package ru.yandex.chemodan.app.dataapi.core.datasources.ydb;

import java.util.function.Function;
import java.util.function.Supplier;

import com.yandex.ydb.core.StatusCode;
import com.yandex.ydb.core.UnexpectedResultException;
import com.yandex.ydb.table.transaction.Transaction;
import com.yandex.ydb.table.transaction.TransactionMode;
import org.joda.time.Instant;
import org.springframework.dao.IncorrectResultSizeDataAccessException;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.SetF;
import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseContext;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.data.record.DatabaseRecord;
import ru.yandex.chemodan.app.dataapi.api.data.snapshot.Snapshot;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceSession;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceType;
import ru.yandex.chemodan.app.dataapi.api.datasource.DsSessionTxManager;
import ru.yandex.chemodan.app.dataapi.api.datasource.ExtendedDataSource;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseDeletionMode;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseExistsException;
import ru.yandex.chemodan.app.dataapi.api.db.filter.DatabasesFilter;
import ru.yandex.chemodan.app.dataapi.api.db.filter.DatabasesFilterOrAll;
import ru.yandex.chemodan.app.dataapi.api.db.filter.DatabasesFilterSource;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandles;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseAliasType;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRefSource;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.internalpublic.PublicDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.revision.DatabaseRefRevisions;
import ru.yandex.chemodan.app.dataapi.api.deltas.DatabaseChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.ModifiedRecordsPojo;
import ru.yandex.chemodan.app.dataapi.api.deltas.OutdatedChangeException;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.CompositeApplicationManager;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.datasources.ydb.dao.DataRecordsYdbDao;
import ru.yandex.chemodan.app.dataapi.core.datasources.ydb.dao.DatabasesYdbDao;
import ru.yandex.chemodan.app.dataapi.core.datasources.ydb.dao.DeletedDatabasesYdbDao;
import ru.yandex.chemodan.app.dataapi.core.datasources.ydb.dao.DeltasYdbDao;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.chemodan.app.dataapi.web.UserNotFoundException;
import ru.yandex.chemodan.ratelimiter.chunk.ChunkRateLimiter;
import ru.yandex.chemodan.ydb.dao.ThreadLocalYdbTransactionManager;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.lang.Validate;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author tolmalev
 */
public class YdbDataSource implements ExtendedDataSource {
    private static final Logger logger = LoggerFactory.getLogger(YdbDataSource.class);
    private static final int QUERY_IN_CLAUSE_SIZE = 1000;

    private final ThreadLocalYdbTransactionManager ydbTransactionManager;
    private final CompositeApplicationManager appManager;
    private final UserMetaManager userMetaManager;

    private final DatabasesYdbDao databasesYdbDao;
    private final DeltasYdbDao deltasYdbDao;
    private final DataRecordsYdbDao dataRecordsYdbDao;
    private final DeletedDatabasesYdbDao deletedDatabasesYdbDao;

    public YdbDataSource(CompositeApplicationManager appManager,
                         UserMetaManager userMetaManager,
                         DataRecordsYdbDao dataRecordsYdbDao,
                         ThreadLocalYdbTransactionManager transactionManager)
    {
        this.appManager = appManager;
        this.userMetaManager = userMetaManager;
        this.dataRecordsYdbDao = dataRecordsYdbDao;
        this.ydbTransactionManager = transactionManager;

        this.databasesYdbDao = new DatabasesYdbDao(transactionManager);
        this.deltasYdbDao = new DeltasYdbDao(transactionManager);
        this.deletedDatabasesYdbDao = new DeletedDatabasesYdbDao(transactionManager);
    }

    @Override
    public DataSourceType type() {
        return DataSourceType.YDB;
    }

    @Override
    public DataSourceSession openSession(UserDatabaseSpec databaseSpec) {
        return new Session(databaseSpec);
    }

    public void deleteDatabases(DataApiUserId uid, DatabasesFilterSource filterSrc) {
        deleteDatabases(uid, filterSrc, DatabaseDeletionMode.MARK_DELETED);
    }

    @Override
    public void deleteDatabases(DataApiUserId uid, DatabasesFilterSource filterSrc, DatabaseDeletionMode mode) {
        deleteDatabases(uid, filterSrc.toDbsFilter(), mode);
    }

    @Override
    public void deleteDatabases(DataApiUserId uid, DatabasesFilterSource filterSrc, DatabaseDeletionMode mode,
                                ChunkRateLimiter rateLimiter) {
        deleteDatabases(uid, filterSrc, mode);
    }

    public void deleteDatabases(DataApiUserId uid, DatabasesFilter filter, DatabaseDeletionMode mode) {
        ydbTransactionManager.executeInTx(() -> {
            ListF<Database> databases = listDatabases(uid, filter.toAllFilter());

            if (mode == DatabaseDeletionMode.MARK_DELETED) {
                moveDatabasesToDeleted(databases, Instant.now());
            } else {
                databasesYdbDao.delete(uid, filter.dbContext(), databases.map(DatabaseRefSource::databaseId));
                cleanupDatabasesData(uid, databases.map(db -> db.dbHandle));
            }
            return null;
        }, TransactionMode.SERIALIZABLE_READ_WRITE);
    }

    public ListF<Database> listDatabases(DataApiUserId uid, DatabasesFilterOrAll filter) {
        switch (filter.getType()) {
            case CONTEXT:
                return databasesYdbDao.find(uid, filter.dbContext());

            case REFS:
                return databasesYdbDao.find(uid, filter.dbContext(), filter.databaseIds());

            case ALL:
                return databasesYdbDao.find(uid);

            default:
                throw new IllegalArgumentException("Unknown database filter specified: " + filter);
        }
    }

    public void moveDatabasesToDeleted(ListF<Database> databases, Instant deletionTime) {
        if (databases.isEmpty()) {
            return;
        }

        Validate.hasSize(1, databases.stableUniqueBy(db -> Tuple2.tuple(db.appNameO(), db.uid)));

        DatabaseContext dbContext = databases.first().dbContext();
        DataApiUserId uid = databases.first().uid;

        ListF<String> databaseIds = databases.map(Database::databaseId);
        try {
            databasesYdbDao.delete(uid, dbContext, databaseIds);
        } catch (IncorrectResultSizeDataAccessException e) {
            // In case, when database not exists
            throw new NotFoundException("Some of databases not found " + dbContext.appNameO() + "." + databaseIds);
        }
        deletedDatabasesYdbDao.saveAsDeletedBatch(uid, databases.map(db -> db.withModificationTime(deletionTime)));
        // records will be removed sometime later in completelyRemoveDatabase()
    }

    @Override
    public Option<Database> getDeletedDatabaseO(DataApiUserId uid, DatabaseHandle dbHandle) {
        return deletedDatabasesYdbDao.find(uid, dbHandle);
    }

    @Override
    public void completelyRemoveDatabasesDeletedBefore(Instant before) {
        int limit = 100;
        ListF<Database> databases;
        int skip = 0;
        do {
            databases = deletedDatabasesYdbDao.findDeletedBefore(before, SqlLimits.range(skip, limit));

            for (Database db : databases) {
                try {
                    completelyRemoveDatabase(db.uid, db.dbHandle);
                } catch (IncorrectResultSizeDataAccessException e) {
                    String dbName = Cf.list(db.uid, db.appNameO(), db.databaseId(), db.handleValue())
                            .mkString(".");
                    logger.error("Failed to remove database {}", dbName, e);
                    skip++;
                }
            }
        } while (databases.size() >= limit);
    }

    @Override
    public ListF<Database> listDatabases(DataApiUserId uid) {
        return listDatabases(uid, DatabasesFilterOrAll.all());
    }

    @Override
    public ListF<Database> listDeletedDatabases(DataApiUserId uid) {
        return deletedDatabasesYdbDao.find(uid);
    }

    @Override
    public ModifiedRecordsPojo getRecordsModifiedSinceRevisions(DataApiUserId uid, DatabaseRefRevisions sinceRevisions,
                                                                Option<Integer> limitO)
    {
        DatabasesFilter filter = sinceRevisions.toDbsFilter()
                .withLimits(limitO.map(limit -> limit + 1)
                        .map(SqlLimits::first)
                        .getOrElse(SqlLimits.all())
                );
        ListF<DatabaseRecord> records = getRecords(uid, filter).map(DataRecord::toDatabaseRecord);
        Option<Integer> takeCountO = limitO.filter(limit -> records.size() > limit);
        return ModifiedRecordsPojo.consFromRecordsSinceRevisions(
                takeCountO.isPresent()
                        ? records.take(takeCountO.get())
                        : records,
                sinceRevisions,
                takeCountO.isPresent()
        );
    }

    private ListF<DataRecord> getRecords(DataApiUserId uid, DatabasesFilter filter) {
        DatabaseHandles dbHandles = getDatabaseHandles(uid, filter);

        if (dbHandles.isEmpty()) {
            return Cf.list();
        }

        return filter.getRevisionsZippedWith(dbHandles)
                .map(revisions ->
                        dataRecordsYdbDao.findByHandlesAndMinRevisionsOrderedByRev(uid, revisions, filter.getLimits()))
                .getOrElse(() ->
                        dataRecordsYdbDao.findByDatabaseHandles(uid, dbHandles));
    }

    private DatabaseHandles getDatabaseHandles(DataApiUserId uid, DatabasesFilter filter) {
        return filter.dbHandlesO()
                .getOrElse(() -> doGetDatabaseHandles(uid, filter));
    }

    private DatabaseHandles doGetDatabaseHandles(DataApiUserId uid, DatabasesFilter filter) {
        return consPermanentDbHandlesO(uid, filter)
                .getOrElse(() -> databasesYdbDao.findHandles(uid, filter))
                .excludeRefs(filter.excludeDbRefs);
    }

    private Option<DatabaseHandles> consPermanentDbHandlesO(DataApiUserId uid, DatabasesFilter filter) {
        return filter.dbRefsSrcO()
                .filterMap(refsSrc -> appManager.consPermanentDbHandlesO(uid, refsSrc));
    }

    private void completelyRemoveDatabase(DataApiUserId uid, DatabaseHandle dbHandle) {
        ydbTransactionManager.executeInTx(() -> {
            deletedDatabasesYdbDao.removeFromDeleted(uid, dbHandle);
            cleanupDatabasesData(uid, Cf.list(dbHandle));
            return null;
        }, TransactionMode.SERIALIZABLE_READ_WRITE);
    }

    private void cleanupDatabasesData(DataApiUserId uid, ListF<DatabaseHandle> dbHandles) {
        ListF<String> handles = dbHandles.map(handle -> handle.handle);
        dataRecordsYdbDao.deleteAllRecordFromDatabases(uid, handles);
        deltasYdbDao.deleteAllForDatabases(uid, handles);
    }

    public class Session implements DataSourceSession {
        private final UserDatabaseSpec userDbRef;
        private Option<Option<Database>> lastObtainedDatabase = Option.empty();
        private final SessionTxManager transactionManager;

        private Option<com.yandex.ydb.table.Session> activeSession;
        private Option<Transaction> activeTransaction;

        Session(UserDatabaseSpec userDbRef) {
            this.userDbRef = userDbRef;
            this.lastObtainedDatabase = userDbRef.databaseO().map(Option::of);
            this.transactionManager = new SessionTxManager(this);
        }

        private void setLastObtainedDatabase(Option<Database> database) {
            lastObtainedDatabase = Option.of(database);
        }

        private void setLastObtainedDatabase(Database database) {
            lastObtainedDatabase = Option.of(Option.of(database));
        }

        private Option<Database> retrieveDatabaseByHandleO(DatabaseHandle dbHandle) {
            return databasesYdbDao.findByHandle(uid(), dbHandle);
        }

        private Option<Database> retrieveDatabaseByRefO() {
            return databasesYdbDao.find(uid(), databaseRef().dbRef());
        }

        public Option<Database> retrieveDatabaseO() {
            Option<Database> database = userDbRef.dbHandleO().isPresent()
                    ? retrieveDatabaseByHandleO(userDbRef.dbHandleO().get())
                    : retrieveDatabaseByRefO();
            setLastObtainedDatabase(database);
            return database;
        }

        @Override
        public UserDatabaseSpec databaseSpec() {
            return userDbRef;
        }

        @Override
        public Option<Database> getDatabaseO() {
            return lastObtainedDatabase.getOrElse(this::retrieveDatabaseO);
        }

        @Override
        public Database createDatabase() {
            return createDatabase(Option.empty());
        }

        @Override
        public Database createDatabaseWithDescription(String title) {
            return createDatabase(Option.of(title));
        }

        private Database createDatabase(Option<String> description) {
            try {
                return createDatabaseInternal(description);
            } catch (UserNotFoundException e) {
                userMetaManager.registerIfNotExists(uid());
                return createDatabaseInternal(description);
            }
        }

        private void saveNewDatabase(Database database) {
            databasesYdbDao.insert(database);
            setLastObtainedDatabase(database);
        }

        private void saveExistingDatabase(Database database, long currentRev) {
            databasesYdbDao.save(database, currentRev);
            setLastObtainedDatabase(database);
        }

        private Database createDatabaseInternal(Option<String> description) {
            Database database = Database.consNew(uid(), generateHandle(), Instant.now(), description);
            saveNewDatabase(database);
            return database;
        }

        private DatabaseHandle generateHandle() {
            return databaseAlias().aliasType() == DatabaseAliasType.PUBLIC
                    ? ((PublicDatabaseAlias) databaseAlias()).consHandle()
                    : appManager.createDatabaseHandle(uid(), databaseRef());
        }

        @Override
        public Database getOrCreateDatabase() {
            return getDatabaseO()
                    .getOrElse(this::safeCreateDatabase);
        }

        private Database safeCreateDatabase() {
            try {
                return createDatabase();
            } catch (DatabaseExistsException e) {
                // XXX: do smth better
                lastObtainedDatabase = Option.empty();
                return getDatabase();
            }
        }

        @Override
        public Database setDatabaseDescription(Option<String> newTitle) {
            Database database = getDatabase()
                    .withDescription(newTitle);
            long currentRev = database.rev;
            saveExistingDatabase(database, currentRev);
            return database;
        }

        @Override
        public void deleteDatabase(DatabaseDeletionMode deletionMode) {
            if (userDbRef.dbHandleO().isPresent()) {
                deleteDatabases(uid(), userDbRef.dbHandleO().get(), deletionMode);
            } else {
                deleteDatabases(uid(), databaseRef(), deletionMode);
            }
        }

        @Override
        public void deleteDatabase(DatabaseDeletionMode deletionMode, ChunkRateLimiter rateLimiter) {
            deleteDatabase(deletionMode);
        }

        @Override
        public Database fixDatabaseRevision(long rev, long currentRev) {
            Database database = getDatabase()
                    .withRev(rev).withModificationTime(Instant.now());

            saveExistingDatabase(database, currentRev);
            return database;
        }

        @Override
        public void onDatabaseUpdate(long rev) {
            throw new UnsupportedOperationException("Could not set revision on native database " + userDbRef);
        }

        @Override
        public Delta getDelta(long rev) {
            return deltasYdbDao.find(getHandle(), rev)
                    .getOrThrow(() -> NotFoundException.consDeltaNotFound(rev));
        }

        @Override
        public ListF<Delta> listDeltas(long fromRev, int limit) {
            return deltasYdbDao.findAfterRevision(getHandle(), fromRev, limit);
        }

        private boolean hasHandleInMemory() {
            return userDbRef.dbHandleO().isPresent()
                    || lastObtainedDatabase.isPresent()
                    || appManager.consPermanentDbHandleO(uid(), databaseRef()).isPresent();
        }

        @Override
        public Option<ListF<DataRecord>> getDataRecordsO(RecordsFilter filter) {
            return getHandleO()
                    .map(handle -> dataRecordsYdbDao.find(uid(),
                            handle,
                            filter.getCollectionIdCond(),
                            filter.getRecordIdCond(),
                            filter.getRecordCond(),
                            filter.getRecordOrder(),
                            filter.limits()
                    ));
        }

        @Override
        public ListF<DataRecord> getDataRecordsByIds(SetF<DataRecordId> recordIds) {
            Database database = getDatabase();
            if (database.rev == 0) {
                return Cf.list();
            }

            ListF<DataRecord> result = recordIds.iterator()
                    .paginate(QUERY_IN_CLAUSE_SIZE)
                    .map(batch -> dataRecordsYdbDao.findRecords(uid(), database.dbHandle, batch))
                    .flatMap(ListF::iterator)
                    .toList();

            return result;
        }

        @Override
        public ListF<DataRecord> getDataRecords(RecordsFilter filter) {
            // Если мы можем получить handle без запроса в базу, делаем это
            // Если нет - идем за объектами одним запросом с вложенным SELECT-ом
            if (hasHandleInMemory()) {
                return getDataRecordsO(filter).getOrElse(Cf.list());
            } else {
                return dataRecordsYdbDao.find(uid(),
                        databaseRef(),
                        filter.getCollectionIdCond(),
                        filter.getRecordIdCond(),
                        filter.getRecordCond(),
                        filter.getRecordOrder(),
                        filter.limits());
            }
        }

        @Override
        public int getDataRecordsCount(RecordsFilter filter) {
            if (hasHandleInMemory()) {
                return getHandleO()
                        .map(handle -> dataRecordsYdbDao.count(uid(),
                                handle,
                                filter.getCollectionIdCond(),
                                filter.getRecordIdCond())
                        )
                        .getOrElse(0);
            } else {
                return dataRecordsYdbDao
                        .count(uid(), databaseRef(), filter.getCollectionIdCond(), filter.getRecordIdCond());
            }
        }

        @Override
        public Option<Snapshot> getSnapshotO(RecordsFilter filter) {
            Option<Database> databaseO = getDatabaseO();
            return databaseO.map(database -> {
                if (database.meta.recordsCount == 0) {
                    return new Snapshot(database, Cf.list());
                }
                // won't fail because handle exists
                return getDataRecordsO(filter)
                        .map(records -> new Snapshot(database, records))
                        .get();
            });
        }

        @Override
        public void save(DatabaseChange change) {
            saveDeltas(change);
            saveRecordChanges(change);
            savePatchedDatabase(change);
        }

        private void saveDeltas(DatabaseChange change) {
            if (change.deltas.isEmpty()) {
                return;
            }

            Database database = change.patchedDatabase();
            try {
                deltasYdbDao.batchInsert(database.uid, database.handleValue(), change.deltas);
            } catch (UnexpectedResultException e) {
                if (e.getStatusCode() == StatusCode.PRECONDITION_FAILED) {
                    throw new OutdatedChangeException(
                            StringUtils.format("Delta with rev={}-{} was already applied for database={}",
                                    change.deltas.first().rev,
                                    change.deltas.last().rev,
                                    database.handleValue()));
                } else {
                    throw e;
                }
            }
        }

        private void saveRecordChanges(DatabaseChange change) {
            dataRecordsYdbDao.bulkInsertDeleteUpdate(uid(), databaseRef(), change.getNewRecords(), change.getDeletedIds(), change.getUpdatedRecords());
        }

        private void savePatchedDatabase(DatabaseChange change) {
            saveExistingDatabase(change.patchedDatabase(), change.sourceDatabase().rev);
        }

        private DatabaseHandle getHandle() {
            return getHandleO()
                    .getOrThrow(userDbRef::consNotFound);
        }

        private Option<DatabaseHandle> getHandleO() {
            return userDbRef.dbHandleO()
                    .orElse(() -> appManager.consPermanentDbHandleO(uid(), databaseRef()))
                    .orElse(() -> getDatabaseO().map(db -> db.dbHandle));
        }

        @Override
        public DsSessionTxManager tx() {
            return transactionManager;
        }
    }

    private class SessionTxManager implements DsSessionTxManager {
        private final Session session;

        private SessionTxManager(Session session) {
            this.session = session;
        }

        @Override
        public DataSourceSession session() {
            return session;
        }

        @Override
        public <R> R executeInReadOnlyTx(Supplier<R> op) {
            return executeInTx(op, TransactionMode.SERIALIZABLE_READ_WRITE);
        }

        @Override
        public <R> R executeInReadCommittedTx(Supplier<R> op) {
            return executeInTx(op, TransactionMode.SERIALIZABLE_READ_WRITE);
        }

        @Override
        public <R> Option<R> executeInTxWithLockedDbIfDbExists(Function<Database, R> op) {
            try {
                return executeInReadCommittedTx(() -> executeWithDbLockedForUpdateO(op));
            } catch(NotFoundException e) {
                return Option.empty();
            }
        }

        private <R> Option<R> executeWithDbLockedForUpdateO(Function<Database, R> function) {
            Option<Database> databaseO = session.retrieveDatabaseO();
            return databaseO.map(function::apply);
        }

        private <T> T executeInTx(Supplier<T> op, TransactionMode mode) {
            return ydbTransactionManager.executeInTx(op, mode);
        }
    }
}
