package ru.yandex.chemodan.app.dataapi.core.datasources.ydb;

import java.util.concurrent.CompletableFuture;

import com.yandex.ydb.core.Result;
import com.yandex.ydb.core.auth.TokenAuthProvider;
import com.yandex.ydb.core.grpc.GrpcTransport;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Profile;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceSessionImplTest;
import ru.yandex.chemodan.app.dataapi.api.datasource.DataSourceType;
import ru.yandex.chemodan.app.dataapi.api.datasource.SessionProviderTestContextConfiguration;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.apps.CompositeApplicationManager;
import ru.yandex.chemodan.app.dataapi.core.dao.support.DataApiRandomValueGenerator;
import ru.yandex.chemodan.app.dataapi.core.dao.usermeta.UserMetaManager;
import ru.yandex.chemodan.app.dataapi.core.datasources.ydb.dao.DataRecordsYdbDao;
import ru.yandex.chemodan.app.dataapi.core.datasources.ydb.dao.DatabasesYdbDao;
import ru.yandex.chemodan.app.dataapi.core.datasources.ydb.dao.DeletedDatabasesYdbDao;
import ru.yandex.chemodan.app.dataapi.core.datasources.ydb.dao.DeltasYdbDao;
import ru.yandex.chemodan.ydb.dao.OneTableYdbDao;
import ru.yandex.chemodan.ydb.dao.ThreadLocalYdbTransactionManager;
import ru.yandex.chemodan.ydb.dao.YdbTimeoutSettings;
import ru.yandex.chemodan.ydb.dao.YdbUtils;
import ru.yandex.chemodan.ydb.dao.pojo.YdbTestUtils;
import ru.yandex.devtools.test.YaTest;

/**
 * @author tolmalev
 */
@Profile("dataapi-ydb-test")
@Configuration
public class YdbDataSourceContextConfiguration {
    @Bean
    public YdbTimeoutSettings timeoutSettings() {
        return YdbTestUtils.getTestTimeoutSettings();
    }

    @Bean
    public YdbDataSource ydbDataSource(CompositeApplicationManager appManager,
                                       UserMetaManager userMetaManager,
                                       DataRecordsYdbDao dataRecordsYdbDao,
                                       ThreadLocalYdbTransactionManager transactionManager)
    {
        return new YdbDataSource(appManager, userMetaManager, dataRecordsYdbDao, transactionManager);
    }

    @Bean
    public DataRecordsYdbDao dataRecordsYdbDao(ThreadLocalYdbTransactionManager transactionManager) {
        return new DataRecordsYdbDao(transactionManager);
    }

    @Bean
    public ThreadLocalYdbTransactionManager ydbTransactionManager(TableClient tableClient,
                                                                  YdbTimeoutSettings timeoutSettings) {
        return new ThreadLocalYdbTransactionManager(tableClient, timeoutSettings);
    }

    @Bean(destroyMethod = "close")
    public TableClient tableClient(YdbTimeoutSettings timeoutSettings) {
        String database;
        String endpoint;

        RpcTransport transport;

        if (YaTest.insideYaTest) {
            endpoint = System.getenv("YDB_ENDPOINT");
            database = System.getenv("YDB_DATABASE");

            transport = GrpcTransport.forEndpoint(endpoint, database).build();
        } else {
            String user = System.getenv("USER");

            endpoint = "ydb-ru-prestable.yandex.net:2135";
            database = "/ru-prestable/home/" + user + "/mydb";

            String token = System.getenv("YDB_TOKEN");
            if (token == null) {
                throw new IllegalStateException("You should set env variable YDB_TOKEN to run tests");
            }
            transport = GrpcTransport.forEndpoint(endpoint, database)
                .withAuthProvider(new TokenAuthProvider(token))
                .build();
        }

        TableClient tableClient = TableClient.newClient(GrpcTableRpc.ownTransport(transport))
                .sessionPoolSize(10, 100)
                .build();
        ThreadLocalYdbTransactionManager transactionManager = new ThreadLocalYdbTransactionManager(tableClient, timeoutSettings);
        ListF<OneTableYdbDao> daos = Cf.list(
                new DatabasesYdbDao(transactionManager),
                new DeletedDatabasesYdbDao(transactionManager),
                new DataRecordsYdbDao(transactionManager),
                new DeltasYdbDao(transactionManager)
        );

        transactionManager.executeInTmpSession((session, txControl) -> {
            YdbUtils.createTablesIfMissing(database, session, daos);
            return CompletableFuture.completedFuture(Result.success(null));
        });

        return tableClient;
    }

    @Bean
    DataSourceSessionImplTest.DataSourceProvider ydbSessionProvider(YdbDataSource ds) {
        DataApiRandomValueGenerator randomValueGenerator = new DataApiRandomValueGenerator();
        return new SessionProviderTestContextConfiguration.RandomRecordDsProvider(
                DataSourceType.YDB,
                () -> new AppDatabaseRef("app", "dbId"),
                ds,
                randomValueGenerator::createDataApiUserId
        );
    }
}
