package ru.yandex.chemodan.app.dataapi.core.datasources.ydb.dao;

import java.time.Instant;

import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;
import org.springframework.dao.IncorrectResultSizeDataAccessException;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.ydb.dao.OneTableYdbDao;
import ru.yandex.chemodan.ydb.dao.ThreadLocalYdbTransactionManager;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;

/**
 * @author tolmalev
 */
public class DeletedDatabasesYdbDao extends OneTableYdbDao {
    public static final TableDescription DESCRIPTION = TableDescription
            .newBuilder()
            .addNonnullColumn("user_id", PrimitiveType.string())
            .addNonnullColumn("app", PrimitiveType.string())
            .addNonnullColumn("dbId", PrimitiveType.string())
            .addNonnullColumn("handle", PrimitiveType.string())
            .addNonnullColumn("rev", PrimitiveType.int64())
            .addNonnullColumn("creation_time", PrimitiveType.timestamp())
            .addNonnullColumn("delete_time", PrimitiveType.timestamp())
            .addNonnullColumn("size", PrimitiveType.int64())
            .addNonnullColumn("records_count", PrimitiveType.int64())
            .addNullableColumn("description", PrimitiveType.string())
            .setPrimaryKeys("user_id", "app", "dbId", "handle")
            .build();

    public static final String TABLE_NAME = "deleted_databases";

    public DeletedDatabasesYdbDao(ThreadLocalYdbTransactionManager transactionManager) {
        super(transactionManager, TABLE_NAME, DESCRIPTION);
    }

    public void saveAsDeletedBatch(DataApiUserId uid, ListF<Database> databases) {
        insertBatch(databases.map(db -> {
            return Tuple2List.<String, Value>fromPairs(
                    "user_id", PrimitiveValue.string(uid.toString().getBytes()),
                    "app", PrimitiveValue.string(db.dbAppId().getBytes()),
                    "dbId", PrimitiveValue.string(db.databaseId().getBytes()),
                    "handle", PrimitiveValue.string(db.handleValue().getBytes()),
                    "rev", PrimitiveValue.int64(db.rev),
                    "creation_time", PrimitiveValue.timestamp(Instant.ofEpochMilli(db.meta.creationTime.getMillis())),
                    "delete_time", PrimitiveValue.timestamp(Instant.ofEpochMilli(db.meta.modificationTime.getMillis())),
                    "size", PrimitiveValue.int64(db.meta.size.toBytes()),
                    "records_count", PrimitiveValue.int64(db.meta.recordsCount),
                    "description", getOptionalStringValue(db.meta.description)
            ).toMap();
        }));
    }

    public Option<Database> find(DataApiUserId uid, DatabaseHandle dbHandle) {
        SqlCondition condition = SqlCondition.trueCondition()
                .and(SqlCondition.column("user_id").eq(uid.toString()))
                .and(SqlCondition.column("handle").eq(dbHandle.handle));

        return queryForList("SELECT * FROM deleted_databases", condition, DatabaseMapper.FROM_DELETED).firstO();
    }

    public ListF<Database> find(DataApiUserId uid) {
        SqlCondition condition = SqlCondition.trueCondition()
                .and(SqlCondition.column("user_id").eq(uid.toString()));

        return queryForList("SELECT * FROM deleted_databases", condition, DatabaseMapper.FROM_DELETED);
    }

    public ListF<Database> findDeletedBefore(org.joda.time.Instant before, SqlLimits range) {
        SqlCondition condition = SqlCondition.trueCondition()
                .and(SqlCondition.column("delete_time").lt(before));

        return queryForList("SELECT * FROM deleted_databases", condition, DatabaseMapper.FROM_DELETED);
    }

    public void removeFromDeleted(DataApiUserId uid, DatabaseHandle dbHandle) {
        String sql = "" +
                "DECLARE $app as String;\n" +
                "DECLARE $user_id as String;\n" +
                "DECLARE $db_id as String;\n" +
                "DECLARE $handle as String;\n" +
                "\n" +
                "SELECT COUNT(*) FROM deleted_databases WHERE user_id = $user_id AND app = $app AND dbId = $db_id AND handle = $handle;\n" +
                "DELETE FROM deleted_databases " +
                "WHERE user_id = $user_id AND app = $app AND dbId = $db_id AND handle = $handle"
                ;

        Params params = Params.create()
                .put("$user_id", PrimitiveValue.string(uid.toString().getBytes()))
                .put("$app", PrimitiveValue.string(dbHandle.dbAppId().getBytes()))
                .put("$db_id", PrimitiveValue.string(dbHandle.databaseId().getBytes()))
                .put("$handle", PrimitiveValue.string(dbHandle.handleValue().getBytes()));

        long affected = queryForLong(sql, params);

        if (affected != 1) {
            throw new IncorrectResultSizeDataAccessException(1, (int) affected);
        }
    }
}
