package ru.yandex.chemodan.app.dataapi.core.datasources.ydb.dao;

import java.time.Instant;

import com.yandex.ydb.table.description.TableDescription;
import com.yandex.ydb.table.query.Params;
import com.yandex.ydb.table.result.ResultSetReader;
import com.yandex.ydb.table.values.ListType;
import com.yandex.ydb.table.values.PrimitiveType;
import com.yandex.ydb.table.values.PrimitiveValue;
import com.yandex.ydb.table.values.Value;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.protobuf.ProtobufDataUtils;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.ydb.dao.OneTableYdbDao;
import ru.yandex.chemodan.ydb.dao.ThreadLocalYdbTransactionManager;
import ru.yandex.chemodan.ydb.dao.YdbRowMapper;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.db.q.SqlOrder;

/**
 * @author tolmalev
 */
public class DeltasYdbDao extends OneTableYdbDao {
    public static final TableDescription DESCRIPTION = TableDescription
            .newBuilder()
            .addNonnullColumn("handle", PrimitiveType.string())
            .addNonnullColumn("time", PrimitiveType.timestamp())
            .addNonnullColumn("rev", PrimitiveType.int64())
            .addNullableColumn("delta_id", PrimitiveType.string())
            .addNonnullColumn("content", PrimitiveType.string())
            .setPrimaryKeys("handle", "rev")
            .build();

    public static final String TABLE_NAME = "deltas";

    public DeltasYdbDao(ThreadLocalYdbTransactionManager transactionManager) {
        super(transactionManager, TABLE_NAME, DESCRIPTION);
    }

    public void deleteAllForDatabases(DataApiUserId uid, ListF<String> handles) {
        String sql = "" +
                "DECLARE $handles as \"List<String>\";" +
                "DELETE from deltas WHERE handle IN $handles";

        ListType listType = ListType.of(PrimitiveType.string());
        Params params = Params.create()
                .put("$handles", listType.newValue(handles.map(h -> PrimitiveValue.string(h.getBytes()))));

        execute(sql, params);
    }

    public Option<Delta> find(DatabaseHandle handle, long rev) {
        String sql = "" +
                "DECLARE $handle as String;\n" +
                "DECLARE $rev as Int64;\n" +
                "\n" +
                "SELECT content from deltas " +
                "WHERE handle = $handle AND rev = $rev";

        Params params = Params.create()
                .put("$handle", PrimitiveValue.string(handle.handle.getBytes()))
                .put("$rev", PrimitiveValue.int64(rev))
                ;

        return queryForList(sql, params, new DeltaMapper()).firstO();
    }

    public ListF<Delta> findAfterRevision(DatabaseHandle handle, long fromRev, int limit) {
        SqlCondition condition = SqlCondition.trueCondition()
                .and(SqlCondition.column("handle").eq(handle.handle))
                .and(SqlCondition.column("rev").ge(fromRev));

        SqlOrder order = SqlOrder.orderByColumn("rev");
        SqlLimits limits = SqlLimits.first(limit);

        return queryForList("SELECT content, rev FROM deltas", condition, order, limits, new DeltaMapper());
    }

    public void insert(DataApiUserId uid, String handle, Delta delta) {
        batchInsert(uid, handle, Cf.list(delta));
    }

    public void batchInsert(DataApiUserId uid, String handle, ListF<Delta> deltas) {
        insertBatch(deltas.map(delta -> Tuple2List.<String, Value>fromPairs(
                "handle", PrimitiveValue.string(handle.getBytes()),
                "rev", PrimitiveValue.int64(delta.rev.get()),
                "delta_id", getOptionalStringValue(delta.id),
                "content", PrimitiveValue.string(serialize(delta)),
                "time", PrimitiveValue.timestamp(Instant.now())
        ).toMap()));
    }

    private static class DeltaMapper implements YdbRowMapper<Delta> {
        @Override
        public Delta mapRow(ResultSetReader rs, int rowNum) {
            return deserialize(rs.getColumn("content").getString());
        }
    }

    private static byte[] serialize(Delta delta) {
        return ProtobufDataUtils.serialize(delta);
    }

    private static Delta deserialize(byte[] serialized) {
        return ProtobufDataUtils.parseDelta(serialized);
    }
}
