package ru.yandex.chemodan.app.dataapi.core.dump;

import com.fasterxml.jackson.databind.JsonNode;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.RecordRef;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.AppDatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.generic.FormatConverterCache;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettings;
import ru.yandex.chemodan.app.dataapi.web.direct.a3.DirectDataApiBenderUtils;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.chemodan.util.tskv.TskvUtils;
import ru.yandex.misc.bender.BenderMapper;
import ru.yandex.misc.bender.annotation.BenderBindAllFields;

/**
 * @author dbrylev
 */
@BenderBindAllFields
public class RecordChangeEvent {
    public static final BenderMapper mapper = DirectDataApiBenderUtils.mapper();
    public static final FormatConverterCache converters = new FormatConverterCache();

    public final DataApiUserId uid;
    public final Option<String> appName;

    public final String databaseId;
    public final String collectionId;
    public final String recordId;
    public final String ycrid;
    public final String appReq;

    public final MethodType method;
    public final long recordRev;
    public final String recordData;

    public final Instant timestamp;

    public RecordChangeEvent(
            DataApiUserId uid, Option<String> appName, String databaseId, String collectionId, String recordId,
            MethodType method, long recordRev, String recordData, String ycrid, String appReq)
    {
        this.uid = uid;
        this.appName = appName;
        this.databaseId = databaseId;
        this.collectionId = collectionId;
        this.recordId = recordId;
        this.method = method;
        this.recordRev = recordRev;
        this.recordData = recordData;
        this.timestamp = Instant.now();
        this.ycrid = ycrid;
        this.appReq = appReq;
    }

    public Tuple2List<String, String> getValues() {
        return appendValues(Tuple2List.arrayList());
    }

    public String formatTskvLine() {
        Tuple2List<String, String> values = Tuple2List.arrayList();

        values.add("tskv_format", DatabaseChangesTskvLogPatternLayout.TSKV_FORMAT);
        values.add("unixtime", Long.toString(System.currentTimeMillis() / 1000));
        values.add("timestamp", Long.toString(timestamp.getMillis()));

        appendValues(values);

        return TskvUtils.formatTskvLine(values.toMap());
    }

    private Tuple2List<String, String> appendValues(Tuple2List<String, String> values) {
        values.add(DatabaseChangesTskvFields.UID, uid.toString());
        values.add(DatabaseChangesTskvFields.APPLICATION_ID, appName.getOrElse(""));
        values.add(DatabaseChangesTskvFields.DATABASE_ID, databaseId);
        values.add(DatabaseChangesTskvFields.COLLECTION_ID, collectionId);
        values.add(DatabaseChangesTskvFields.METHOD, method.getValue());
        values.add(DatabaseChangesTskvFields.RECORD_ID, recordId);
        values.add(DatabaseChangesTskvFields.RECORD_REV, String.valueOf(recordRev));

        if (method != MethodType.DELETE) {
            values.add(DatabaseChangesTskvFields.RECORD_DATA, recordData);
        }
        values.add(DatabaseChangesTskvFields.YCRID, String.valueOf(ycrid));
        values.add(DatabaseChangesTskvFields.APP_REQ, String.valueOf(appReq));
        return values;
    }

    public DatabaseRef getDbRef() {
        return AppDatabaseRef.cons(appName, databaseId);
    }

    public RecordRef getRecordRef() {
        return getDbRef().consColRef(collectionId).consRecordRef(recordId);
    }

    public static RecordChangeEvent cons(
            MethodType method, Database database, DataRecord record, Option<TypeSettings> type,
            String ycrid, String appReq)
    {
        return new RecordChangeEvent(
                database.uid, database.appNameO(), database.databaseId(), record.getCollectionId(), record.getRecordId(),
                method, record.getRev(), getRecordData(record, type), ycrid, appReq);
    }

    private static String getRecordData(DataRecord record, Option<TypeSettings> type) {
        if (type.isPresent()) {
            return converters.getConverter(type.get()).toJson(record.getData()
                    .plus1(type.get().idPropertyName, DataField.string(record.getRecordId())));
        }
        String serializedRecord = new String(mapper.serializeJson(record.toSnapshotPojoRow()));
        JsonNode jsonRecord = JsonNodeUtils.getNode(serializedRecord);
        return jsonRecord.get("fields").toString();
    }
}
