package ru.yandex.chemodan.app.dataapi.core.generic;

import java.util.UUID;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import org.joda.time.Instant;

import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.OrderedUUID;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.DatabaseAccessType;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabasesRegistry;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.DeltaUpdateOrDeleteNonExistentRecordException;
import ru.yandex.chemodan.app.dataapi.api.deltas.FieldChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChangeType;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.generic.bazinga.GenericObjectDeletionTask;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.Condition;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.FilterBuildingContext;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.ObjectsFilter;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.DataRecordsFilter;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.DataRecordsLoader;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.FormatConverter;
import ru.yandex.chemodan.app.dataapi.web.AccessForbiddenException;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Denis Bakharev
 */
public class GenericObjectManager {
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final DataApiManager dataApiManager;
    private final TypeSettingsRegistry typeSettingsRegistry;
    private final FormatConverterCache formatConverterCache;
    private final BazingaTaskManager bazingaTaskManager;
    private final DataRecordsLoader dataRecordsLoader;
    private final ExternalDatabasesRegistry externalDatabasesRegistry;

    public GenericObjectManager(
            DataApiManager dataApiManager,
            TypeSettingsRegistry typeSettingsRegistry,
            FormatConverterCache formatConverterCache,
            BazingaTaskManager bazingaTaskManager,
            DataRecordsLoader dataRecordsLoader,
            ExternalDatabasesRegistry externalDatabasesRegistry)
    {
        this.dataApiManager = dataApiManager;
        this.typeSettingsRegistry = typeSettingsRegistry;
        this.formatConverterCache = formatConverterCache;
        this.bazingaTaskManager = bazingaTaskManager;
        this.dataRecordsLoader = dataRecordsLoader;
        this.externalDatabasesRegistry = externalDatabasesRegistry;
    }

    public void checkAccess(Option<String> clientApp, String typeName, boolean write) {
        if (!clientApp.isPresent()) {
            throw new AccessForbiddenException("Client " + clientApp + " has no access to type " + typeName);
        }
        TypeSettings ts = typeSettingsRegistry.getTypeSettings(typeName);

        DatabaseRef dbRef = ts.dbRef();
        ExternalDatabaseAlias alias =
                new ExternalDatabaseAlias(clientApp.get(), dbRef.appNameO().get(), dbRef.databaseId());

        Option<DatabaseAccessType> accessType = externalDatabasesRegistry.getExternalDatabaseAccessType(alias);
        if (!accessType.isPresent()) {
            throw new AccessForbiddenException("Client " + clientApp + " has no access to type " + typeName);
        }

        if (write && !accessType.isSome(DatabaseAccessType.READ_WRITE)) {
            throw new AccessForbiddenException("Client " + clientApp + " has no write access to type " + typeName);
        }
    }

    public String set(DataApiUserId uid, String objectId, String typeName, String jsonData) {
        return set(uid, objectId, typeName, jsonData, Condition.trueCondition());
    }

    public String set(DataApiUserId uid, String objectId, String typeName, String jsonData, Condition rewriteIf) {
        TypeSettings ts = typeSettingsRegistry.getTypeSettings(typeName);

        if (!ts.isAllowSet) {
            throw new MethodNotAllowedException();
        }

        logger.info("set generic object with uid={}; objectId={}; typename={}", uid, objectId, typeName);

        return saveAndReturnResult(uid, jsonData, ts, objectId, RecordChangeType.SET, rewriteIf);
    }

    public String insert(DataApiUserId uid, String typeName, String jsonData) {

        TypeSettings ts = typeSettingsRegistry.getTypeSettings(typeName);

        if (!ts.isAllowInsert) {
            throw new MethodNotAllowedException();
        }

        String objectId = ts.orderedId.get() ? OrderedUUID.generateOrderedUUID() : UUID.randomUUID().toString();
        logger.info("insert generic object with uid={}; objectId={}; typename={}", uid, objectId, typeName);

        return saveAndReturnResult(uid, jsonData, ts, objectId, RecordChangeType.INSERT, Condition.trueCondition());
    }

    private String saveAndReturnResult(
            DataApiUserId uid, String jsonData, TypeSettings ts,
            String objectId, RecordChangeType recordChangeType, Condition rewriteIf)
    {
        MapF<String, DataField> dataFields = getDataFields(jsonData, ts);
        dataFields.removeTs(ts.idPropertyName);

        ListF<FieldChange> fieldChanges = dataFields.entries().map(FieldChange::put);
        RecordChange recordChange = new RecordChange(
                recordChangeType, ts.typeLocation.collectionId, objectId, fieldChanges);

        Database database = dataApiManager.getOrCreateDatabase(new UserDatabaseSpec(uid, ts.dbRef()));

        dataApiManager.applyDelta(database, recordChange, rewriteIf.buildCondition(
                new FilterBuildingContext(ts, formatConverterCache.getConverter(ts), Option.of(dataFields))));

        return getJson(dataFields, objectId, ts);
    }

    public String get(DataApiUserId uid, String recordId, String typeName) {
        TypeSettings ts = typeSettingsRegistry.getTypeSettings(typeName);

        DataRecord record = dataRecordsLoader.getRecord(uid, ts.typeLocation, recordId);

        return getJson(record.getData(), recordId, ts);
    }

    public LimitedResult<String> getList(DataApiUserId uid, String typeName, ObjectsFilter filter, boolean requestTotalCount) {
        TypeSettings ts = typeSettingsRegistry.getTypeSettings(typeName);

        DataRecordsFilter recsFilter = filter.buildFilter(new FilterBuildingContext(
                ts, formatConverterCache.getConverter(ts), Option.empty()));

        if (!requestTotalCount) {
            recsFilter = recsFilter.withoutTotalCount();
        }

        LimitedResult<DataRecord> records = dataRecordsLoader.getRecords(uid, ts.typeLocation, recsFilter);

        ListF<String> jsonResults = records.result.map(r -> getJson(r.getData(), r.getRecordId(), ts));
        return new LimitedResult<>(records.totalCount, jsonResults, recsFilter.getLimits());
    }

    public void delete(DataApiUserId uid, String recordId, String typeName) {
        logger.info("delete generic object with uid={}; objectId={}; typename={}", uid, recordId, typeName);
        TypeSettings ts = typeSettingsRegistry.getTypeSettings(typeName);

        Database db = dataApiManager.getDatabase(new UserDatabaseSpec(uid, ts.dbRef()));
        try {
            dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD,
                    new Delta(RecordChange.delete(ts.typeLocation.collectionId, recordId)));

        } catch (DeltaUpdateOrDeleteNonExistentRecordException e) {
            throw new NotFoundException(e.getMessage());
        }
    }

    private MapF<String, DataField> getDataFields(String jsonData, TypeSettings typeSettings) {
        FormatConverter formatConverter = formatConverterCache.getConverter(typeSettings);
        try {
            return formatConverter.toDataFields(jsonData);
        } catch (RuntimeException e) {
            if(e.getCause() instanceof JsonParseException || e.getCause() instanceof JsonMappingException) {
                throw new InvalidInputException("Invalid json = " + jsonData, e.getCause());
            }
            throw e;
        }
    }

    private String getJson(MapF<String, DataField> dataFields, String recordId, TypeSettings typeSettings) {
        FormatConverter formatConverter = formatConverterCache.getConverter(typeSettings);
        dataFields = dataFields.plus1(typeSettings.idPropertyName, DataField.string(recordId));

        return formatConverter.toJson(dataFields);
    }

    public void deleteByDeletionDate(DataApiUserId uid, String recordId, String typeName) {
        logger.info(
                "deleteByDeletionDate generic object with uid={}; objectId={}; typename={}", uid, recordId, typeName);

        TypeSettings ts = typeSettingsRegistry.getTypeSettings(typeName);
        Option<DataRecord> recordO = dataApiManager.getRecord(uid, ts.toColRef().consRecordRef(recordId));
        if (ts.deletionSettings.isPresent() && recordO.isPresent()) {
            Option<Instant> deletionDateO =
                    GenericObjectUtils.calculateDeletionDate(recordO.get().getData(), ts.deletionSettings.get());

            if (deletionDateO.isPresent()) {
                Instant deletionDate = deletionDateO.get();
                //проверяем еще раз дату удаления, т.к ее значение могло измениться например из-за
                // измений в TypeSettings или изменений в самом объекте

                if (deletionDate.isBeforeNow()) {
                    delete(uid, recordId, typeName);
                } else {
                    logger.info(
                            "reschedule generic object deletion with uid={}; objectId={}; typename={}; to {}",
                            uid,
                            recordId,
                            typeName,
                            deletionDate);

                    bazingaTaskManager.schedule(new GenericObjectDeletionTask(uid, typeName, recordId), deletionDate);
                }
            }
        }
    }

}
