package ru.yandex.chemodan.app.dataapi.core.generic;

import org.joda.time.Instant;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;
import org.mockito.verification.VerificationMode;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.test.context.ContextConfiguration;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.datasource.SessionProviderTestContextConfiguration;
import ru.yandex.chemodan.app.dataapi.api.db.Database;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.db.ref.external.ExternalDatabaseAlias;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppDatabaseSettings;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.dao.NotFoundHandler;
import ru.yandex.chemodan.app.dataapi.core.dao.data.DataRecordsJdbcDao;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.datasources.DataSourceTypeRegistry;
import ru.yandex.chemodan.app.dataapi.core.generic.bazinga.GenericObjectDeletionTask;
import ru.yandex.chemodan.app.dataapi.core.generic.filter.ObjectsFilter;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.chemodan.app.dataapi.web.AccessForbiddenException;
import ru.yandex.chemodan.app.dataapi.web.DatabaseNotFoundException;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.misc.dataSize.DataSize;
import ru.yandex.misc.db.q.SqlCondition;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.test.Assert;

/**
 * @author tolmalev
 */
@ContextConfiguration(classes = GenericObjectManagerTestBase.Context.class)
@ActivateDataApiEmbeddedPg
public class GenericObjectManagerTestBase extends DataApiTestSupport {

    @Autowired
    protected GenericObjectManager genericObjectManager;

    @Autowired
    protected TypeSettingsRegistry typeSettingsRegistry;

    @Autowired
    protected DataApiManager dataApiManager;

    @Autowired
    protected DataRecordsJdbcDao recordsDao;

    @Autowired
    protected DataSourceTypeRegistry dataSourceTypeRegistry;

    protected DataApiUserId uid;
    protected final Instant now = Instant.now();
    protected static final TypeSettings ts = getTypeSettings();
    protected final String recId = "recId";

    public static final ExternalDatabaseAlias RO_DB_ALIAS =
            new ExternalDatabaseAlias("generic_client_ro", ts.dbRef().appNameO().get(), ts.dbRef().databaseId());

    public static final ExternalDatabaseAlias RW_DB_ALIAS =
            new ExternalDatabaseAlias("generic_client_rw", ts.dbRef().appNameO().get(), ts.dbRef().databaseId());

    @Before
    public void before() {
        typeSettingsRegistry.setTypeSettings(ts);
        uid = createRandomCleanUser();
        Mockito.clearInvocations(recordsDao);
    }

    @Test
    public void deleteByDeletionDate_DeletionDateBeforeNow_DeleteRecord() {

        createObjectWithDateAndTryToDelete(now.minus(ts.deletionSettings.get().getDeletionInterval()));

        Assert.isEmpty(getRecord(recId));
    }

    @Test
    public void deleteRecord() {
        genericObjectManager.set(uid, recId, ts.typeName, "{\"requiredProp\":\"value\"}");

        Assert.some(getRecord(recId));

        genericObjectManager.delete(uid, recId, ts.typeName);

        Assert.none(getRecord(recId));
    }

    @Test
    public void deleteNonExistent() {
        Assert.Block deleteNonExistent = () -> genericObjectManager.delete(uid, "non-existent", ts.typeName);

        Assert.assertThrows(deleteNonExistent, DatabaseNotFoundException.class);

        dataApiManager.createDatabase(new UserDatabaseSpec(uid, ts.dbRef()));

        Assert.assertThrows(deleteNonExistent, NotFoundException.class);
    }

    @Test
    public void createObject_DoNotStoreObjectIdAsDataField() {
        String json = "{\"requiredProp\":\"value\"}";
        genericObjectManager.set(uid, recId, ts.typeName, json);

        DataRecord record = getRecord(recId).get();
        Assert.equals(1, record.getData().size());
        Assert.equals(DataField.string("value"), record.getData().getTs("requiredProp"));

        assertDbCallsCount();
    }

    private void createObjectWithDateAndTryToDelete(Instant dateValue) {
        MapF<String, DataField> data = createRequiredGenericObjectData(recId, "someValue");
        data.put("date", DataField.timestamp(dateValue));

        Database db = dataApiManager.createDatabase(new UserDatabaseSpec(uid, ts.dbRef()));
        dataApiManager.applyDelta(db, RevisionCheckMode.PER_RECORD,
                new Delta(RecordChange.insert(ts.typeLocation.collectionId, recId, data)));

        Assert.notEmpty(getRecord(recId));
        assertDbCallsCount();

        genericObjectManager.deleteByDeletionDate(uid, recId, ts.typeName);
    }

    @Test
    public void deleteByDeletionDate_DeletionDateAfterNow_RescheduleTask() {
        createObjectWithDateAndTryToDelete(now);

        Assert.notEmpty(getRecord(recId));
        assertDbCallsCount();

        Assert.isInstance(bazingaStub.tasksWithParams.get(0), GenericObjectDeletionTask.class);
    }



    @Test
    public void getList_limitsAll_oneDbCall() {
        String json = "{\"requiredProp\":\"value\"}";

        Cf.range(0, 10).forEach(i -> genericObjectManager.set(uid, "id" + i, ts.typeName, json));

        genericObjectManager.getList(uid, ts.typeName, ObjectsFilter.all(), true);

        assertDbCallsCount();
    }

    @Test
    public void getList_limitsRange_oneDbCall() {
        String json = "{\"requiredProp\":\"value\"}";

        Cf.range(0, 9).forEach(i -> genericObjectManager.set(uid, "id" + i, ts.typeName, json));

        genericObjectManager.getList(uid, ts.typeName, ObjectsFilter.all().withLimits(SqlLimits.range(0, 10)), true);

        assertDbCallsCount();
    }

    @Test
    public void getList_limitsRange_oneCountDbCalls() {
        String json = "{\"requiredProp\":\"value\"}";

        Cf.range(0, 10).forEach(i -> genericObjectManager.set(uid, "id" + i, ts.typeName, json));

        genericObjectManager.getList(uid, ts.typeName, ObjectsFilter.all().withLimits(SqlLimits.range(0, 10)), true);

        assertDbCallsCount2(Mockito.calls(1));
    }

    @Test
    public void getList_limitsRange_no_request_count_oneDbCall() {
        String json = "{\"requiredProp\":\"value\"}";

        Cf.range(0, 10).forEach(i -> genericObjectManager.set(uid, "id" + i, ts.typeName, json));

        genericObjectManager.getList(uid, ts.typeName, ObjectsFilter.all().withLimits(SqlLimits.range(0, 10)), false);

        assertDbCallsCount2(Mockito.never());
    }

    @Test(expected = AccessForbiddenException.class)
    public void checkAccess_noClient_read_throwException() {
        genericObjectManager.checkAccess(Option.empty(), ts.typeName, false);
    }

    @Test(expected = AccessForbiddenException.class)
    public void checkAccess_noClient_write_throwException() {
        genericObjectManager.checkAccess(Option.empty(), ts.typeName, true);
    }

    @Test(expected = AccessForbiddenException.class)
    public void checkAccess_hasClient_noAccess_read_throwException() {
        genericObjectManager.checkAccess(Option.of("client_no_access"), ts.typeName, false);
    }

    @Test(expected = AccessForbiddenException.class)
    public void checkAccess_hasClient_noAccess_write_throwException() {
        genericObjectManager.checkAccess(Option.of("client_no_access"), ts.typeName, true);
    }

    @Test
    public void checkAccess_hasClient_read_access_read_ok() {
        genericObjectManager.checkAccess(Option.of("generic_client_ro"), ts.typeName, false);
    }

    @Test(expected = AccessForbiddenException.class)
    public void checkAccess_hasClient_read_access_write_throwException() {
        genericObjectManager.checkAccess(Option.of("generic_client_ro"), ts.typeName, true);
    }

    @Test
    public void checkAccess_hasClient_write_access_read_ok() {
        genericObjectManager.checkAccess(Option.of("generic_client_rw"), ts.typeName, false);
    }

    @Test
    public void checkAccess_hasClient_write_access_write_ok() {
        genericObjectManager.checkAccess(Option.of("generic_client_rw"), ts.typeName, true);
    }

    private Option<DataRecord> getRecord(String recId) {
        return dataApiManager.getRecord(uid, ts.toColRef().consRecordRef(recId));
    }

    protected void assertDbCallsCount() {
        Mockito
                .inOrder(recordsDao)
                .verify(recordsDao, Mockito.calls(1))
                .find(
                        Mockito.any(),
                        Mockito.any(DatabaseRef.class),
                        Mockito.any(), Mockito.any(), Mockito.any(), Mockito.any(), Mockito.any()
                );

        Mockito
                .inOrder(recordsDao)
                .verify(recordsDao, Mockito.never())
                .count(Mockito.any(), Mockito.any(SqlCondition.class));
    }

    protected void assertDbCallsCount2(VerificationMode calls) {
        Mockito
                .inOrder(recordsDao)
                .verify(recordsDao, Mockito.calls(1))
                .find(
                        Mockito.any(),
                        Mockito.any(DatabaseRef.class),
                        Mockito.any(), Mockito.any(), Mockito.any(), Mockito.any(), Mockito.any()
                );

        Mockito
                .inOrder(recordsDao)
                .verify(recordsDao, calls)
                .count(Mockito.any(), Mockito.any(DatabaseRef.class), Mockito.any(), Mockito.any());
    }

    @ContextConfiguration
    static class Context {
        @Bean
        @Primary
        NotFoundHandler.ProxyCreator notFoundHandlerProxyCreator() {
            return SessionProviderTestContextConfiguration.notFoundHandlerProxyCreator;
        }

        @Bean
        @Primary
        AppSettingsRegistry appSettingsRegistry() {
            AppSettingsRegistry mock = Mockito.mock(AppSettingsRegistry.class);
            AppDatabaseSettings def = new AppDatabaseSettings(Option.empty(), Option.empty());

            def.setDatabaseSizeLimit(Option.of(DataSize.MEGABYTE));

            Mockito.when(mock.getDatabaseSettings(Mockito.any())).thenReturn(def);
            Mockito.when(mock.getDefaultDatabasesCountLimit()).thenReturn(100);
//            Mockito.when(mock.getSettings(Mockito.any())).thenReturn(def);

            return mock;
        }
    }
}
