package ru.yandex.chemodan.app.dataapi.core.generic;

import java.util.Iterator;
import java.util.concurrent.ConcurrentHashMap;

import com.fasterxml.jackson.databind.JsonNode;
import com.github.fge.jsonschema.core.report.LogLevel;
import com.github.fge.jsonschema.core.report.ProcessingMessage;
import com.github.fge.jsonschema.core.report.ProcessingReport;
import com.github.fge.jsonschema.main.JsonSchema;
import com.github.fge.jsonschema.main.JsonSchemaFactory;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.FormatConverter;
import ru.yandex.chemodan.util.exception.SchemaValidationException;
import ru.yandex.chemodan.util.json.JsonNodeUtils;
import ru.yandex.misc.ExceptionUtils;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Denis Bakharev
 */
public class GenericObjectsValidator extends GenericObjectChangedEventHandler {

    private static final Logger logger = LoggerFactory.getLogger(GenericObjectsValidator.class);

    private final FormatConverterCache formatConverterCache;
    private final ConcurrentHashMap<String, JsonSchema> jsonSchemaValidators = new ConcurrentHashMap<>();

    public GenericObjectsValidator(
            TypeSettingsRegistry typeSettingsRegistry, FormatConverterCache formatConverterCache)
    {
        super(typeSettingsRegistry);
        this.formatConverterCache = formatConverterCache;
    }

    public ListF<RecordChange> handleGenericObjectRecordChanged(
            Option<DataRecord> recordAtStart,
            DataRecord recordAtEnd,
            TypeSettings typeSettings)
    {
        FormatConverter converter = formatConverterCache.getConverter(typeSettings);
        MapF<String, DataField> dataFields = recordAtEnd.getData();
        dataFields = dataFields.plus1(typeSettings.idPropertyName, DataField.string(recordAtEnd.getRecordId()));
        try {
            String json = converter.toJson(dataFields);
            validateByJsonSchema(json, typeSettings, recordAtEnd.getRecordId());
        } catch (Exception e) {
            logger.error("Schema validation failed: {}, {} typeSettings: {}",
                    dataFields, recordAtEnd.getRecordId(), typeSettings, e);
            if (!typeSettings.isDryRun) {
                throw e;
            }
        }
        return Cf.list();
    }

    private void validateByJsonSchema(String json, TypeSettings typeSettings, String recordId) {
        JsonSchema validator =
                jsonSchemaValidators.computeIfAbsent(typeSettings.jsonSchema, this::computeSchemaValidatorObject);

        ProcessingReport report = validator.validateUnchecked(JsonNodeUtils.getNode(json));

        if (!report.isSuccess()) {
            throw new SchemaValidationException(makeExceptionString(report, recordId));
        }
    }

    private String makeExceptionString(ProcessingReport report, String objectId) {
        Iterator<ProcessingMessage> iterator = report.iterator();
        StringBuilder sb = new StringBuilder();
        sb.append(StringUtils.format("[ObjectId={}]", objectId));
        while (iterator.hasNext()) {
            ProcessingMessage processingMessage = iterator.next();
            if (processingMessage.getLogLevel() == LogLevel.ERROR
                || processingMessage.getLogLevel() == LogLevel.FATAL)
            {
                sb.append("\n");
                sb.append(processingMessage.getMessage());
            }
        }

        return sb.toString();
    }

    private JsonSchema computeSchemaValidatorObject(String jsonSchema) {
        try {
            JsonNode jsonNode = JsonNodeUtils.getNode(jsonSchema);
            return JsonSchemaFactory.byDefault().getJsonSchema(jsonNode);
        } catch (Exception e) {
            throw ExceptionUtils.translate(e);
        }
    }

    @Override
    public int getOrder() {
        return VALIDATION_ORDER;
    }
}
