package ru.yandex.chemodan.app.dataapi.core.generic;

import org.joda.time.Instant;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.deltas.RecordChange;
import ru.yandex.chemodan.app.dataapi.core.generic.bazinga.GenericObjectDeletionTask;
import ru.yandex.commune.bazinga.BazingaTaskManager;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Denis Bakharev
 */
public class ScheduleObjectDeletionHandler extends GenericObjectChangedEventHandler {
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final BazingaTaskManager bazingaTaskManager;

    public ScheduleObjectDeletionHandler(
            BazingaTaskManager bazingaTaskManager,
            TypeSettingsRegistry typeSettingsRegistry)
    {
        super(typeSettingsRegistry);
        this.bazingaTaskManager = bazingaTaskManager;
    }

    @Override
    public ListF<RecordChange> handleGenericObjectRecordChanged(
            Option<DataRecord> recordAtStart, DataRecord recordAtEnd, TypeSettings ts)
    {
        if (ts.deletionSettings.isPresent()) {
            DeletionSettings ds = ts.deletionSettings.get();
            if (!recordAtStart.isPresent() || isDeletionDateChanged(recordAtStart.get(), recordAtEnd, ds)) {

                Option<Instant> deletionDateO = GenericObjectUtils.calculateDeletionDate(recordAtEnd.getData(), ds);
                if (deletionDateO.isPresent()) {
                    logger.info(
                            "schedule generic object deletion with uid={}; objectId={}; typename={}; to {}",
                            recordAtEnd.uid,
                            recordAtEnd.getRecordId(),
                            ts.typeName,
                            deletionDateO.get());

                    GenericObjectDeletionTask task =
                            new GenericObjectDeletionTask(recordAtEnd.uid, ts.typeName, recordAtEnd.getRecordId());
                    bazingaTaskManager.schedule(task, deletionDateO.get());
                }
            }
        }

        return Cf.list();
    }

    private boolean isDeletionDateChanged(DataRecord recordsAtStart, DataRecord recordAtEnd, DeletionSettings ds) {
        Option<Instant> deletionDateAtStart = GenericObjectUtils.calculateDeletionDate(recordsAtStart.getData(), ds);
        Option<Instant> deletionDateAtEnd = GenericObjectUtils.calculateDeletionDate(recordAtEnd.getData(), ds);
        return !deletionDateAtStart.equals(deletionDateAtEnd);
    }

    @Override
    public int getOrder() {
        return SCHEDULE_TASKS_ORDER;
    }
}

