package ru.yandex.chemodan.app.dataapi.core.generic.filter;

import java.util.function.BiFunction;

import lombok.Data;
import lombok.EqualsAndHashCode;
import org.jparsec.OperatorTable;
import org.jparsec.Parser;

import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordCondition;

/**
 * @author dbrylev
 */
public abstract class Condition {

    static Parser<?> tokenizer() {
        return ConditionParser.terminals.tokenizer().cast().or(Predicate.tokenizer());
    }

    static Parser<? extends Condition> parser() {
        Parser<? extends Predicate> predicate = Predicate.parser();

        Parser.Reference<Condition> ref = Parser.newReference();

        Parser<Condition> unit = ref.lazy()
                .between(ConditionParser.terminals.token("("), ConditionParser.terminals.token(")"))
                .or(predicate);

        Parser<Condition> parser = new OperatorTable<Condition>()
                .prefix(ConditionParser.terminals.token("not").retn(Not::new), 30)
                .infixl(ConditionParser.terminals.token("and").retn(Binary.consF("and")), 20)
                .infixl(ConditionParser.terminals.token("or").retn(Binary.consF("or")), 10)
                .build(unit);

        ref.set(parser);
        return parser;
    }

    public static Condition trueCondition() {
        return new True();
    }

    public abstract RecordCondition buildCondition(FilterBuildingContext context);

    public static class True extends Condition {
        public RecordCondition buildCondition(FilterBuildingContext context) {
            return RecordCondition.all();
        }
    }

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class Binary extends Condition {
        private final Condition left;
        private final String op;
        private final Condition right;

        public static BiFunction<Condition, Condition, Binary> consF(String op) {
            return (left, right) -> new Binary(left, op, right);
        }

        @Override
        public RecordCondition buildCondition(FilterBuildingContext context) {
            switch (op) {
                case "and": return left.buildCondition(context).and(right.buildCondition(context));
                case "or": return left.buildCondition(context).or(right.buildCondition(context));
            }
            throw new IllegalStateException("Unexpected operator " + op);
        }
    }

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class Not extends Condition {
        private final Condition condition;

        @Override
        public RecordCondition buildCondition(FilterBuildingContext context) {
            return condition.buildCondition(context).not();
        }
    }
}
