package ru.yandex.chemodan.app.dataapi.core.generic.filter;

import org.joda.time.Instant;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordCondition;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeLocation;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettings;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.FormatConverter;
import ru.yandex.misc.test.Assert;
import ru.yandex.misc.time.MoscowTime;

/**
 * @author dbrylev
 */
public class ConditionParserTest {

    @Test
    public void expression() {
        String expr = "NOT (A>=1 or not b != True and c between '1' and 2.56)"
                + " or k not in ($y, 2) And (l is not null or not m)";

        Predicate a = new Predicate.Comparison(new Field("A"), ">=", new Value.Numeric(1L));
        Predicate b = new Predicate.Comparison(new Field("b"), "!=", new Value.Bool(true));

        Predicate c = new Predicate.Between(new Field("c"), new Value.Text("1"), new Value.Numeric(2.56d), true);

        Condition abc = new Condition.Binary(a, "or", new Condition.Binary(new Condition.Not(b), "and", c));

        ListF<Value> set = Cf.list(new Value.Placeholder("y"), new Value.Numeric(2L));
        Predicate k = new Predicate.InSet(new Field("k"), set, false);

        Predicate l = new Predicate.IsNull(new Field("l"), false);
        Predicate m = new Predicate.BoolValue(new Field("m"));

        Condition klm = new Condition.Binary(k, "and", new Condition.Binary(l, "or", new Condition.Not(m)));

        Assert.equals(new Condition.Binary(new Condition.Not(abc), "or", klm), ConditionParser.parser.parse(expr));
    }


    private static final FormatConverter format = new FormatConverter("{"
            + "  \"type\":\"object\","
            + "  \"properties\": {"
            + "    \"bool\": {\"type\": \"boolean\"},"
            + "    \"time\": {\"type\": \"string\", \"dataapi-convert-type\": \"timestamp\"},"
            + "    \"num\": {\"type\": \"number\"},"
            + "    \"int\": {\"type\": \"integer\"},"
            + "    \"str\": {\"type\": \"string\"},"
            + "    \"id\": {\"type\": \"string\"}"
            + "  }"
            + "}");

    private static final TypeSettings type = new TypeSettings(
            "", "", "id", false, false, Cf.list(Order.empty()), Mockito.mock(TypeLocation.class), Option.empty(), false);

    @Test
    public void booleanPredicate() {
        Assert.isTrue(matches("bool", Cf.map("bool", DataField.bool(true))));
        Assert.isFalse(matches("not bool", Cf.map("bool", DataField.bool(true))));

        Assert.isFalse(matches("bool", Cf.map("bool", DataField.bool(false))));
        Assert.isTrue(matches("not bool", Cf.map("bool", DataField.bool(false))));

        Assert.isFalse(matches("bool", Cf.map()));

        Assert.assertThrows(() -> parse("int"), FilterBuildingException.class);
    }

    @Test
    public void booleanIsNull() {
        Assert.isFalse(matches("bool is null", Cf.map("bool", DataField.bool(true))));
        Assert.isTrue(matches("bool is null", Cf.map()));

        Assert.isTrue(matches("bool is not null", Cf.map("bool", DataField.bool(false))));
        Assert.isFalse(matches("bool is not null", Cf.map()));

        Assert.isFalse(matches("not bool", Cf.map()));
        Assert.isTrue(matches("not bool is not null", Cf.map()));
    }

    @Test
    public void timestampComparison() {
        Instant ts = Instant.now();
        String time = "'" + ts.toString() + "'";

        Assert.isTrue(matches("time =" + time, Cf.map("time", DataField.timestamp(ts))));
        Assert.isFalse(matches(time + "!= time", Cf.map("time", DataField.timestamp(ts))));

        Assert.isTrue(matches("time > " + time, Cf.map("time", DataField.timestamp(ts.plus(1)))));
        Assert.isFalse(matches(time + " >= time", Cf.map("time", DataField.timestamp(ts.plus(1)))));

        Assert.isFalse(matches("time < " + time, Cf.map("time", DataField.timestamp(ts.plus(1)))));
        Assert.isTrue(matches(time + " <= time", Cf.map("time", DataField.timestamp(ts.plus(1)))));

        Assert.isTrue(matches("time = '2018-02-21'",
                Cf.map("time", DataField.timestamp(MoscowTime.instant(2018, 2, 21, 0, 0))))); // ¯\_(ツ)_/¯

        Assert.assertThrows(() -> parse("time = 1519245552404"), FilterBuildingException.class);
    }

    @Test
    public void numberInSet() {
        Assert.isTrue(matches("num in (1, 1.75)", Cf.map("num", DataField.decimal(1))));
        Assert.isTrue(matches("num in (2, 2.75)", Cf.map("num", DataField.decimal(2.75))));

        Assert.isFalse(matches("num in (3, 3.75)", Cf.map("num", DataField.decimal(55))));
        Assert.isFalse(matches("num in (4, 3.75)", Cf.map()));

        Assert.isTrue(matches("num not in (3)", Cf.map("num", DataField.decimal(0))));
        Assert.isTrue(matches("not num in (3)", Cf.map("num", DataField.decimal(0))));
        Assert.isFalse(matches("not num in (3)", Cf.map()));

        Assert.assertThrows(() -> parse("num in ('1')"), FilterBuildingException.class);
    }

    @Test
    public void integerBetween() {
        Assert.isTrue(matches("int between 1 and 2", Cf.map("int", DataField.integer(1))));
        Assert.isTrue(matches("int between 1 and 2", Cf.map("int", DataField.integer(2))));

        Assert.isFalse(matches("int between 1 and 2", Cf.map("int", DataField.integer(3))));
        Assert.isTrue(matches("int not between 1 and 2", Cf.map("int", DataField.integer(3))));
        Assert.isTrue(matches("not int between 1 and 2", Cf.map("int", DataField.integer(3))));

        Assert.isTrue(matches("not int between 1 and 2", Cf.map("int", DataField.integer(3))));
        Assert.isFalse(matches("not int between 1 and 2", Cf.map()));

        Assert.assertThrows(() -> parse("int between 1 and 2.5"), FilterBuildingException.class);
        Assert.assertThrows(() -> parse("int between 1 and '2'"), FilterBuildingException.class);
    }

    @Test
    public void stringPlaceholder() {
        FilterBuildingContext context = new FilterBuildingContext(
                type, format, Option.of(Cf.map("value", DataField.string("string"))));

        Assert.isTrue(matches("str = $value", Cf.map("str", DataField.string("string")), context));
        Assert.isFalse(matches("str = $value", Cf.map("str", DataField.string("nope")), context));

        Assert.isTrue(matches("str < $value", Cf.map("str", DataField.string("nope")), context));
        Assert.isFalse(matches("str < $value", Cf.map(), context));

        Assert.assertThrows(() -> parse("num = $value", context), FilterBuildingException.class);
        Assert.assertThrows(() -> parse("str != $missing", context), FilterBuildingException.class);
    }

    @Test
    public void recordIdCondition() {
        Assert.isTrue(matches("id != 'record' or num = 0", Cf.map("num", DataField.decimal(0))));
        Assert.isFalse(matches("id != 'record' or num = 0", Cf.map()));

        Assert.isTrue(matches("id = 'record' and num != 0", Cf.map("num", DataField.decimal(1))));
        Assert.isFalse(matches("id = 'record' and num != 0", Cf.map()));

        Assert.assertThrows(() -> parse("id = id"), FilterBuildingException.class);
        Assert.assertThrows(() -> parse("'record' = 'record'"), FilterBuildingException.class);
    }


    private static FilterBuildingContext consContext(Option<MapF<String, DataField>> data) {
        return new FilterBuildingContext(type, format, data);
    }

    private static final FilterBuildingContext context = consContext(Option.empty());

    private static DataRecord consRecord(MapF<String, DataField> data) {
        return new DataRecord(new DataApiPassportUserId(1),
                new DataRecordId(DatabaseHandle.consGlobal("db", "handle"), "collection", "record"), 0L, data);
    }

    private static RecordCondition parse(String expression) {
        return parse(expression, context);
    }

    private static boolean matches(String expression, MapF<String, DataField> data) {
        return matches(expression, data, context);
    }

    private static RecordCondition parse(String expression, FilterBuildingContext context) {
        return ConditionParser.parse(expression, context);
    }

    private static boolean matches(String expression, MapF<String, DataField> data, FilterBuildingContext context) {
        return parse(expression, context).matches(consRecord(data));
    }
}
