package ru.yandex.chemodan.app.dataapi.core.generic.filter;

import lombok.Data;
import org.jparsec.Parser;
import org.jparsec.Parsers;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByDataRecordOrder;
import ru.yandex.misc.bender.annotation.BenderTextValue;

/**
 * @author dbrylev
 */
@Data
public class Order {

    private final ListF<Component> components;

    static Parser<?> tokenizer() {
        return OrderParser.terminals.tokenizer().cast().or(Component.tokenizer());
    }

    static Parser<Order> parser() {
        return Component.parser().sepBy1(OrderParser.terminals.token(",")).map(cs -> new Order(Cf.toList(cs)));
    }

    public static Order empty() {
        return new Order(Cf.list());
    }

    public boolean isNotEmpty() {
        return components.isNotEmpty();
    }

    public ByDataRecordOrder buildOrder(FilterBuildingContext context) {
        return components.iterator().map(c -> c.buildOrder(context))
                .reduceLeftO(ByDataRecordOrder::andThen)
                .getOrElse(ByDataRecordOrder::unordered);
    }

    public Order reversed() {
        return new Order(components.map(c -> new Component(c.field, !c.asc)));
    }

    public boolean startsWith(Order prefix) {
        IteratorF<Component> componentsIt = components.iterator();

        return prefix.components.forAll(c -> componentsIt.nextO().isSome(c));
    }

    @BenderTextValue
    public static Order benderParse(String value) {
        return value.equals("*") ? empty() : OrderParser.parse(value);
    }

    @BenderTextValue
    public String benderSerialize() {
        return components.isNotEmpty()
                ? components.map(c -> (c.isAsc() ? "" : "-") + c.field.getName()).mkString(", ")
                : "*";
    }

    @Data
    public static class Component {

        private final Field field;
        private final boolean asc;

        public static Parser<?> tokenizer() {
            return Field.tokenizer();
        }

        public static Parser<Component> parser() {
            Parser<Boolean> positive = OrderParser.terminals.token("+").map(t -> true);
            Parser<Boolean> negative = OrderParser.terminals.token("-").map(t -> false);

            return Parsers.sequence(
                    Parsers.or(positive, negative).optional(true),
                    Field.parser(), (sign, field) -> new Component(field, sign));
        }

        public ByDataRecordOrder buildOrder(FilterBuildingContext context) {
            return field.getColumn(context).orderBy(asc);
        }
    }
}
