package ru.yandex.chemodan.app.dataapi.core.generic.filter;

import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.MapF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.bolts.collection.Tuple2List;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.filter.ordering.ByDataRecordOrder;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiPassportUserId;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeLocation;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeSettings;
import ru.yandex.chemodan.app.dataapi.utils.dataconversion.FormatConverter;
import ru.yandex.misc.test.Assert;

/**
 * @author dbrylev
 */
public class OrderParserTest {

    private static final FormatConverter format = new FormatConverter("{"
            + "  \"type\":\"object\","
            + "  \"properties\": {"
            + "    \"num\": {\"type\": \"number\"},"
            + "    \"id\": {\"type\": \"string\"}"
            + "  }"
            + "}");

    private static final TypeSettings type = new TypeSettings(
            "", "", "id", false, false, Cf.list(), Mockito.mock(TypeLocation.class), Option.empty(), false);

    @Test
    public void parsing() {
        Assert.assertThrows(() -> parse("   "), FilterBuildingException.class);
        Assert.assertThrows(() -> parse("unknown"), FilterBuildingException.class);
        Assert.assertThrows(() -> parse("$num"), FilterBuildingException.class);
        Assert.assertThrows(() -> parse("1"), FilterBuildingException.class);
    }

    @Test
    public void ordering() {
        assertsSorted("num, -id", Cf.list("rec1", "rec3", "rec2"), Tuple2List.fromPairs(
                "rec3", Cf.map("num", DataField.decimal(5)),
                "rec2", Cf.map("num", DataField.decimal(5)),
                "rec1", Cf.map("num", DataField.decimal(1))));

        assertsSorted("-num, -id", Cf.list("rec3", "rec2", "rec1"), Tuple2List.fromPairs(
                "rec1", Cf.map("num", DataField.decimal(1)),
                "rec2", Cf.map("num", DataField.decimal(5)),
                "rec3", Cf.map("num", DataField.decimal(5))));

        assertsSorted("-num, id", Cf.list("rec1", "rec3", "rec2"), Tuple2List.fromPairs(
                "rec1", Cf.map(),
                "rec2", Cf.map("num", DataField.decimal(0)),
                "rec3", Cf.map()));
    }

    @Test
    public void bending() {
        Assert.equals("f1, -f2", Order.benderParse("+f1, -f2").benderSerialize());
        Assert.equals("-f1, f2", Order.benderParse("f1, -f2").reversed().benderSerialize());

        Assert.equals("*", Order.benderParse("*").benderSerialize());
        Assert.isFalse(Order.benderParse("*").isNotEmpty());
    }

    private static ByDataRecordOrder parse(String expression) {
        return OrderParser.parse(expression, new FilterBuildingContext(type, format, Option.empty()));
    }

    private static void assertsSorted(
            String expression, ListF<String> expectedRecordIds, Tuple2List<String, MapF<String, DataField>> records)
    {
        ByDataRecordOrder order = parse(expression);

        Assert.equals(expectedRecordIds,
                records.map(OrderParserTest::consRecord).sorted(order.comparator()).map(DataRecord::getRecordId));
    }

    private static DataRecord consRecord(String recordId, MapF<String, DataField> data) {
        return new DataRecord(new DataApiPassportUserId(1),
                new DataRecordId(DatabaseHandle.consGlobal("db", "handle"), "collection", recordId), 0L, data);
    }
}
