package ru.yandex.chemodan.app.dataapi.core.generic.filter;

import lombok.Data;
import lombok.EqualsAndHashCode;
import org.jparsec.Parser;
import org.jparsec.Parsers;
import org.jparsec.Terminals;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.ConvertingRecordColumn;
import ru.yandex.chemodan.app.dataapi.api.data.filter.condition.RecordCondition;

/**
 * @author dbrylev
 */
public abstract class Predicate extends Condition {

    public static Parser<?> tokenizer() {
        return Operand.tokenizer();
    }

    public static Parser<? extends Predicate> parser() {
        return Parsers.or(
                Comparison.parser(),
                IsNull.parser(),
                InSet.parser(),
                Between.parser(),
                BoolValue.parser());
    }

    public abstract Field getField();

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class Comparison extends Predicate {
        private static final ListF<String> ops = Cf.list("=", "!=", ">", ">=", "<", "<=");

        private final Operand left;
        private final String op;
        private final Operand right;

        public static Parser<Comparison> parser() {
            Parser<? extends Operand> operand = Operand.parser();

            return Parsers.or(ops.map(
                    op -> Parsers.sequence(
                            operand, ConditionParser.terminals.token(op), operand,
                            (left, token, right) -> new Comparison(left, op, right))));
        }

        private <T extends Operand> T getOperand(Class<T> type) {
            if (type.isInstance(left)) {
                return type.cast(left);
            }
            if (type.isInstance(right)) {
                return type.cast(right);
            }
            throw new FilterBuildingException("Missing " + type.getSimpleName().toLowerCase()
                    + " within " + Cf.list(left.valueString(), op, right.valueString()).mkString(" "));
        }

        @Override
        public RecordCondition buildCondition(FilterBuildingContext context) {
            Field field = getOperand(Field.class);
            Value value = getOperand(Value.class);

            ConvertingRecordColumn<Value> column = field.getColumn(context);

            switch (op) {
                case "=": return column.eq(value);
                case "!=": return column.ne(value);
                case ">": return field == left ? column.gt(value) : column.lt(value);
                case "<": return field == left ? column.lt(value) : column.gt(value);
                case ">=": return field == left ? column.ge(value) : column.le(value);
                case "<=": return field == left ? column.le(value) : column.ge(value);
            }
            throw new IllegalStateException("Unexpected operator: " + op);
        }

        @Override
        public Field getField() {
            return getOperand(Field.class);
        }
    }

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class BoolValue extends Predicate {

        private final Field field;

        public static Parser<BoolValue> parser() {
            return Field.parser().map(BoolValue::new);
        }

        @Override
        public RecordCondition buildCondition(FilterBuildingContext context) {
            return field.getColumn(context).eq(new Value.Bool(true));
        }

        public Field getField() {
            return field;
        }
    }

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class IsNull extends Predicate {

        private final Field field;
        private final boolean positive;

        public static Parser<IsNull> parser() {
            Terminals terminals = ConditionParser.terminals;

            return Parsers.sequence(
                    Field.parser(), terminals.token("is"),
                    terminals.token("not").optional(null), terminals.token("null"),
                    (fld, is, not, nil) -> new IsNull(fld, not == null));
        }

        @Override
        public RecordCondition buildCondition(FilterBuildingContext context) {
            return positive ? field.getColumn(context).isNull() : field.getColumn(context).isNotNull();
        }

        public Field getField() {
            return field;
        }
    }

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class InSet extends Predicate {

        private final Field field;
        private final ListF<? extends Value> values;
        private final boolean positive;

        public static Parser<InSet> parser() {
            Terminals terminals = ConditionParser.terminals;

            Parser<Field> field = Field.parser();
            Parser<? extends Value> value = Value.parser();

            return Parsers.sequence(
                    field, terminals.token("not").optional(null), terminals.token("in"),
                    terminals.token("("), value.sepBy1(terminals.token(",")), terminals.token(")"),
                    (fld, not, in, ob, values, cb) -> new InSet(fld, Cf.x(values), not == null));
        }

        @Override
        public RecordCondition buildCondition(FilterBuildingContext context) {
            RecordCondition base = field.getColumn(context).inSet(values);

            return positive ? base : base.not();
        }

        public Field getField() {
            return field;
        }
    }

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class Between extends Predicate {

        private final Field field;
        private final Value min;
        private final Value max;
        private final boolean positive;

        public static Parser<Between> parser() {
            Terminals terminals = ConditionParser.terminals;

            Parser<Field> field = Field.parser();
            Parser<? extends Value> value = Value.parser();

            return Parsers.sequence(
                    field, terminals.token("not").optional(null), terminals.token("between"),
                    value, terminals.token("and"), value,
                    (fld, not, between, min, and, max) -> new Between(fld, min, max, not == null));
        }

        @Override
        public RecordCondition buildCondition(FilterBuildingContext context) {
            RecordCondition base = field.getColumn(context).between(min, max);

            return positive ? base : base.not();
        }

        public Field getField() {
            return field;
        }
    }
}
