package ru.yandex.chemodan.app.dataapi.core.generic.filter;

import lombok.Data;
import lombok.EqualsAndHashCode;
import org.jparsec.Parser;
import org.jparsec.Parsers;
import org.jparsec.Terminals;
import org.jparsec.Tokens;
import org.jparsec.pattern.Patterns;

import ru.yandex.misc.db.q.SqlQueryUtils;
import ru.yandex.misc.lang.StringUtils;

/**
 * @author dbrylev
 */
public abstract class Value extends Operand {

    public static Parser<?> tokenizer() {
        return Parsers.or(
                Numeric.tokenizer(), Text.tokenizer(),
                Bool.tokenizer(), Placeholder.tokenizer());
    }

    public static Parser<? extends Value> parser() {
        return Parsers.or(
                Numeric.parser(), Text.parser(),
                Bool.parser(), Placeholder.parser());
    }

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class Bool extends Value {

        private final boolean value;

        public static Parser<?> tokenizer() {
            return Parsers.never();
        }

        public static Parser<Bool> parser() {
            Terminals terminals = ConditionParser.terminals;

            return Parsers.or(
                    terminals.token("true").map(t -> new Bool(true)),
                    terminals.token("false").map(t -> new Bool(false)));
        }

        @Override
        public String valueString() {
            return Boolean.toString(value);
        }
    }

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class Numeric extends Value {

        private final Number value;

        public static Parser<?> tokenizer() {
            return Terminals.DecimalLiteral.TOKENIZER;
        }

        public static Parser<Numeric> parser() {
            return Terminals.DecimalLiteral.PARSER.map(s -> s.contains(".")
                    ? new Numeric(Double.parseDouble(s))
                    : new Numeric(Long.parseLong(s)));
        }

        public boolean isFractional() {
            return value.doubleValue() != value.longValue();
        }

        @Override
        public String valueString() {
            return isFractional() ? Double.toString(value.doubleValue()) : Long.toString(value.longValue());
        }
    }

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class Text extends Value {

        private final String value;

        public static Parser<?> tokenizer() {
            return Terminals.StringLiteral.SINGLE_QUOTE_TOKENIZER;
        }

        public static Parser<Text> parser() {
            return Terminals.StringLiteral.PARSER.map(Text::new);
        }

        @Override
        public String valueString() {
            return SqlQueryUtils.quote(value);
        }
    }

    @Data
    @EqualsAndHashCode(callSuper = false)
    public static class Placeholder extends Value {

        private static final String tag = "PLACEHOLDER";

        private final String field;

        public static Parser<?> tokenizer() {
            Parser<String> scanner = Patterns.isChar('$').next(Patterns.WORD).toScanner("placeholder").source();

            return scanner.map(v -> Tokens.fragment(StringUtils.substringAfter(v, "$"), tag));
        }

        public static Parser<Placeholder> parser() {
            return Terminals.fragment(tag).map(Placeholder::new);
        }

        @Override
        public String valueString() {
            return "$" + field + " value";
        }
    }

}
