package ru.yandex.chemodan.app.dataapi.core.generic.filter;

import org.joda.time.Instant;

import ru.yandex.chemodan.app.dataapi.api.data.field.DataField;
import ru.yandex.chemodan.app.dataapi.api.data.field.DataFieldType;

/**
 * @author dbrylev
 */
public abstract class ValueConverter<T> {

    public static ValueConverter<Long> integer(FilterBuildingContext context) {
        return new ValueConverter<Long>(DataFieldType.INTEGER, context) {

            @Override
            protected Long doConvert(Value.Numeric value) {
                validate(!value.isFractional(), value);
                return value.getValue().longValue();
            }

            @Override
            protected Long doConvert(DataField data) {
                return data.integerValue();
            }
        };
    }

    public static ValueConverter<Double> decimal(FilterBuildingContext context) {
        return new ValueConverter<Double>(DataFieldType.DECIMAL,context) {

            @Override
            protected Double doConvert(Value.Numeric value) {
                return value.getValue().doubleValue();
            }

            @Override
            protected Double doConvert(DataField data) {
                return data.decimalValue();
            }
        };
    }

    public static ValueConverter<Boolean> bool(FilterBuildingContext context) {
        return new ValueConverter<Boolean>(DataFieldType.BOOLEAN, context) {
            @Override
            protected Boolean doConvert(Value.Bool value) {
                return value.isValue();
            }

            @Override
            protected Boolean doConvert(DataField data) {
                return data.booleanValue();
            }
        };
    }

    public static ValueConverter<String> string(FilterBuildingContext context) {
        return new ValueConverter<String>(DataFieldType.STRING, context) {

            @Override
            protected String doConvert(Value.Text value) {
                return value.getValue();
            }

            @Override
            protected String doConvert(DataField data) {
                return data.stringValue();
            }
        };
    }

    public static ValueConverter<Instant> timestamp(FilterBuildingContext context) {
        return new ValueConverter<Instant>(DataFieldType.TIMESTAMP, context) {
            @Override
            protected Instant doConvert(Value.Text value) {
                try {
                    return Instant.parse(value.getValue());

                } catch (IllegalArgumentException e) {
                    throw conversionException(value);
                }
            }

            @Override
            protected Instant doConvert(DataField data) {
                return data.timestampValue();
            }
        };
    }

    private final DataFieldType resultType;
    private final FilterBuildingContext context;

    private ValueConverter(DataFieldType resultType, FilterBuildingContext context) {
        this.resultType = resultType;
        this.context = context;
    }

    public T convert(Value value) {
        if (value instanceof Value.Bool) {
            return doConvert((Value.Bool) value);
        }
        if (value instanceof Value.Numeric) {
            return doConvert((Value.Numeric) value);
        }
        if (value instanceof Value.Text) {
            return doConvert((Value.Text) value);
        }
        if (value instanceof Value.Placeholder) {
            DataField data = context.getPlaceholderDataField((Value.Placeholder) value);

            validate(data.fieldType == resultType, value);
            return doConvert(data);
        }
        throw new IllegalStateException("Unexpected value type: " + value.getClass());
    }

    protected FilterBuildingException conversionException(Value value) {
        throw new FilterBuildingException(
                "Can not convert " + value.valueString() + " to " + resultType.name().toLowerCase());
    }

    protected void validate(boolean condition, Value value) {
        if (!condition) {
            throw conversionException(value);
        }
    }

    protected T doConvert(Value.Bool value) {
        throw conversionException(value);
    }

    protected T doConvert(Value.Numeric value) {
        throw conversionException(value);
    }

    protected T doConvert(Value.Text value) {
        throw conversionException(value);
    }

    protected abstract T doConvert(DataField data);
}
