package ru.yandex.chemodan.app.dataapi.core.generic.loader;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.handle.DatabaseHandle;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.generic.LimitedResult;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeLocation;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata.AppDataLoader;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata.model.AppCollectionData;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata.model.AppData;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata.model.AppDatabaseData;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata.model.AppId;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata.model.AppRecordData;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.commune.dynproperties.DynamicProperty;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Denis Bakharev
 * @author Maksim Ahramovich
 */
public class DataRecordsFromAppDataLoader implements DataRecordsLoader {
    private final DynamicProperty<Boolean> enableAppDataCache =
            new DynamicProperty<>("dataapi-enable-appdata-cache", true);
    private final Logger logger = LoggerFactory.getLogger(getClass());
    private final AppDataLoader appDataLoader;
    private final AppSettingsRegistry appSettingsRegistry;
    private final DataRecordsLoader downstreamLoader;

    public DataRecordsFromAppDataLoader(
            AppDataLoader appDataLoader,
            AppSettingsRegistry appSettingsRegistry,
            DataRecordsLoader downstreamLoader)
    {
        this.appDataLoader = appDataLoader;
        this.appSettingsRegistry = appSettingsRegistry;
        this.downstreamLoader = downstreamLoader;
    }

    @Override
    public DataRecord getRecord(DataApiUserId uid, TypeLocation typeLocation, String recordId) {
        Option<AppData> appDataO = loadAppData(uid, typeLocation);
        if (!appDataO.isPresent()) {
            return downstreamLoader.getRecord(uid, typeLocation, recordId);
        }

        AppData appData = appDataO.get();
        Option<DataRecord> recordDataO = appData.getDatabasesMap().getO(typeLocation.databaseId).flatMapO(
                dbData -> dbData.getCollectionsMap().getO(typeLocation.collectionId).flatMapO(
                        colData -> colData.getRecordsMap().getO(recordId).flatMapO(
                                recData -> Option.of(getDataRecord(uid, appData, dbData, colData, recData)))));

        return recordDataO.getOrThrow(NotFoundException.consF("Record not found: " + recordId));
    }

    @Override
    public LimitedResult<DataRecord> getRecords(DataApiUserId uid, TypeLocation typeLocation, DataRecordsFilter filter) {
        Option<AppData> appDataO = loadAppData(uid, typeLocation);
        if (!appDataO.isPresent()) {
            return downstreamLoader.getRecords(uid, typeLocation, filter);
        }

        AppData appData = appDataO.get();
        Option<LimitedResult<DataRecord>> resultO =
                appData.getDatabasesMap().getO(typeLocation.databaseId).flatMapO(
                        dbData -> dbData.getCollectionsMap().getO(typeLocation.collectionId).flatMapO(
                                colData -> Option.of(getLimitedResult(uid, appData, dbData, colData, filter))
                        )
                );

        return resultO.getOrElse(new LimitedResult<>(Option.of(0), Cf.list(), filter.getLimits()));
    }

    private LimitedResult<DataRecord> getLimitedResult(
            DataApiUserId uid,
            AppData appData,
            AppDatabaseData dbData,
            AppCollectionData colData,
            DataRecordsFilter filter)
    {
        ListF<DataRecord> dataRecords = getRecords(uid, appData, dbData, colData);

        if (!filter.getCondition().isAll()) {
            dataRecords = dataRecords.filter(filter.getCondition()::matches);
        }
        if (filter.getOrder().isPresent()) {
            dataRecords = dataRecords.sorted(filter.getOrder().get().comparator());
        }

        return new LimitedResult<>(
                Option.when(filter.isTotalCountNeeded(), colData.getRecordsMap().size()),
                filter.getLimits().get(dataRecords), filter.getLimits());
    }

    private ListF<DataRecord> getRecords(
            DataApiUserId uid,
            AppData appData,
            AppDatabaseData dbData,
            AppCollectionData colData)
    {
        return colData.getRecordsMap().values().map(recData -> getDataRecord(uid, appData, dbData, colData, recData));
    }

    private DataRecord getDataRecord(
            DataApiUserId uid, AppData appData, AppDatabaseData dbData, AppCollectionData colData,
            AppRecordData recData)
    {
        DataRecordId recordId = getRecordId(appData, dbData, colData, recData);
        return new DataRecord(uid, recordId, recData.getRevision(), recData.getDataFieldMap());
    }

    private DataRecordId getRecordId(AppData appData, AppDatabaseData dbData, AppCollectionData colData,
            AppRecordData recData)
    {
        DatabaseHandle handle =
                DatabaseRef.cons(appData.getId().getAppName(), dbData.getDatabaseId())
                        .consHandle(dbData.getDatabaseHandle());
        return new DataRecordId(handle, colData.getCollectionId(), recData.getRecordId());
    }

    private Option<AppData> loadAppData(DataApiUserId uid, TypeLocation typeLocation) {
        if(!enableAppDataCache.get()) {
            return Option.empty();
        }

        if(!isCacheableTypeLocation(typeLocation)) {
            return Option.empty();
        }

        return appDataLoader.loadAppData(new AppId(uid, typeLocation.app));
    }

    private boolean isCacheableTypeLocation(TypeLocation typeLocation) {
        return appSettingsRegistry.getDatabaseSettings(typeLocation.dbRef()).isUseCache();
    }
}
