package ru.yandex.chemodan.app.dataapi.core.generic.loader;

import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.TransactionStatus;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.CollectionRef;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.support.ShardedTransactionManager;
import ru.yandex.chemodan.app.dataapi.core.generic.LimitedResult;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeLocation;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.web.NotFoundException;
import ru.yandex.chemodan.app.dataapi.web.UserNotFoundException;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

/**
 * @author Denis Bakharev
 */
public class DataRecordsFromDatabaseLoader implements DataRecordsLoader {
    private final static Logger logger = LoggerFactory.getLogger(DataRecordsFromDatabaseLoader.class);

    private final DataApiManager dataApiManager;
    private final ShardedTransactionManager transactionManager;

    public DataRecordsFromDatabaseLoader(DataApiManager dataApiManager, ShardedTransactionManager transactionManager) {
        this.dataApiManager = dataApiManager;
        this.transactionManager = transactionManager;
    }

    @Override
    public DataRecord getRecord(DataApiUserId uid, TypeLocation tl, String recordId) {
        return dataApiManager.getRecord(uid, tl.toColRef().consRecordRef(recordId))
                .getOrThrow(NotFoundException.consF("Record not found: " + recordId));
    }

    protected LimitedResult<DataRecord> getRecordsRaw(
            DataApiUserId uid, TypeLocation typeLocation, DataRecordsFilter filter)
    {
        SqlLimits limits = filter.getLimits();

        TransactionStatus transaction = null;
        //we need transaction only for limited request - to get count in the same transaction
        if (filter.isTotalCountNeeded()) {
            transaction = startReadOnlyTransaction(uid);
        }

        try {
            CollectionRef colRef = typeLocation.toColRef();
            UserDatabaseSpec databaseSpec = new UserDatabaseSpec(uid, colRef.dbRef());

            ListF<DataRecord> records = dataApiManager.getRecords(databaseSpec, filter.toRecordsFilter(colRef));

            int totalCount = records.size();
            if (filter.isTotalCountNeeded()) {
                if (!records.isEmpty() && records.size() < limits.getCount()) {
                    totalCount = limits.getFirst() + records.size();
                } else {
                    totalCount = dataApiManager.getRecordsCount(databaseSpec, RecordsFilter.DEFAULT.withColRef(colRef));
                }
            }

            return new LimitedResult<>(Option.when(filter.isTotalCountNeeded(), totalCount), records, limits);
        } finally {
            if (transaction != null) {
                transactionManager.rollback(uid, transaction);
            }
        }
    }

    @Override
    public LimitedResult<DataRecord> getRecords(DataApiUserId uid, TypeLocation typeLocation, DataRecordsFilter filter) {
        try {
            return getRecordsRaw(uid, typeLocation, filter);
        } catch (UserNotFoundException e) {
            logger.warn("User not found {} {}", uid, e);
            return new LimitedResult<>(Option.empty(), Cf.list(), filter.getLimits());
        }
    }

    private TransactionStatus startReadOnlyTransaction(DataApiUserId uid) {
        DefaultTransactionDefinition definition = new DefaultTransactionDefinition();
        definition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        definition.setReadOnly(true);
        return transactionManager.getTransaction(uid, definition);
    }
}
