package ru.yandex.chemodan.app.dataapi.core.generic.loader;

import org.easymock.Mock;
import org.junit.Before;
import org.junit.Test;
import org.mockito.Mockito;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.ListF;
import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.data.filter.RecordsFilter;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecord;
import ru.yandex.chemodan.app.dataapi.api.data.record.DataRecordId;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.core.dao.support.ShardedTransactionManager;
import ru.yandex.chemodan.app.dataapi.core.generic.LimitedResult;
import ru.yandex.chemodan.app.dataapi.core.generic.TypeLocation;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.web.ActionUtils;
import ru.yandex.misc.db.q.SqlLimits;
import ru.yandex.misc.test.Assert;

import static org.mockito.Matchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.only;

/**
 * @author akirakozov
 */
public class DataRecordsFromDatabaseLoaderTest {

    private static final DataApiUserId UID = DataApiUserId.parse("123");
    private static final TypeLocation TYPE_LOCATION = new TypeLocation(Option.empty(), "dbId", "collId");

    @Mock
    private DataApiManagerMock dataApiManager;
    @Mock
    private ShardedTransactionManager transactionManager;

    private DataRecordsFromDatabaseLoader dataRecordsLoader;

    @Before
    public void init() {
        dataApiManager = Mockito.mock(DataApiManagerMock.class, Mockito.CALLS_REAL_METHODS);
        dataApiManager.init(95);
        transactionManager = Mockito.mock(ShardedTransactionManager.class);
        dataRecordsLoader = new DataRecordsFromDatabaseLoader(dataApiManager, transactionManager);
    }

    @Test
    public void getAllRecords() {
        LimitedResult<DataRecord> result = dataRecordsLoader.getRecords(
                UID, TYPE_LOCATION, DataRecordsFilter.all());

        checkResult(result, Option.empty(), 95, Option.of("record_0"), Option.of("record_94"), false, false);
    }

    @Test
    public void getFirstTwelveRecords() {
        LimitedResult<DataRecord> result = dataRecordsLoader.getRecords(
                UID, TYPE_LOCATION, DataRecordsFilter.limited(SqlLimits.first(12)));
        checkResult(result, 95, 12, "record_0", "record_11", true, true);
    }

    @Test
    public void getNineFromThirdRecords() {
        LimitedResult<DataRecord> result = dataRecordsLoader.getRecords(
                UID, TYPE_LOCATION, DataRecordsFilter.limited(SqlLimits.range(3, 9)));
        checkResult(result, 95, 9, "record_3", "record_11", true, true);
    }

    @Test
    public void getFirstRecordsWithBigLimit() {
        LimitedResult<DataRecord> result = dataRecordsLoader.getRecords(
                UID, TYPE_LOCATION, DataRecordsFilter.limited(SqlLimits.first(100)));
        checkResult(result, 95, 95, "record_0", "record_94", false, true);
    }

    @Test
    public void getFirstRecordsWithOffsetAndBigLimit() {
        LimitedResult<DataRecord> result = dataRecordsLoader.getRecords(
                UID, TYPE_LOCATION, DataRecordsFilter.limited(SqlLimits.range(11, 100)));
        checkResult(result, 95, 84, "record_11", "record_94", false, true);
    }

    @Test
    public void getRecordsWithOffsetGreaterThanCount() {
        LimitedResult<DataRecord> result = dataRecordsLoader.getRecords(
                UID, TYPE_LOCATION, DataRecordsFilter.limited(SqlLimits.range(200, 10)));
        checkResult(result, Option.of(95), 0, Option.empty(), Option.empty(), true, true);
    }

    @Test
    public void getRecordsWithOffsetWithoutLimit() {
        LimitedResult<DataRecord> result = dataRecordsLoader.getRecords(
                UID, TYPE_LOCATION, DataRecordsFilter.limited(ActionUtils.getLimits(Option.empty(), Option.of(20))));
        checkResult(result, 95, 75, "record_20", "record_94", false, true);
    }

    private void checkResult(LimitedResult<DataRecord> result, int totalCount, int resultCount,
            String firstId, String lastId, boolean shouldCallCount, boolean shouldStartTransaction)
    {
        checkResult(result, Option.of(totalCount), resultCount,
                Option.of(firstId), Option.of(lastId), shouldCallCount, shouldStartTransaction);
    }

    private void checkResult(LimitedResult<DataRecord> result, Option<Integer> totalCount, int resultCount,
            Option<String> firstId, Option<String> lastId, boolean shouldCallCount, boolean shouldStartTransaction)
    {
        Assert.equals(totalCount, result.totalCount, "Incorrect total records count");
        Assert.equals(resultCount, result.result.size(), "Incorrect records number in result");
        if (firstId.isPresent() && lastId.isPresent()) {
            Assert.equals(firstId.get(), result.result.first().getRecordId(), "Incorrect first record id");
            Assert.equals(lastId.get(), result.result.last().getRecordId(), "Incorrect last record id");
        } else {
            Assert.isEmpty(result.result, "Results list should be empty");
        }

        // Check optimization, that count request is not called
        if (!shouldCallCount) {
            Mockito.verify(dataApiManager, never()).getRecordsCount(any(), any());
        }

        if (!shouldStartTransaction) {
            Mockito.verify(transactionManager, never()).getTransaction(any(), any());
            Mockito.verify(transactionManager, never()).getTransaction(any());
        } else {
            Mockito.verify(transactionManager, only()).getTransaction(any(), any());
        }
    }

    // Mock object only to check limits
    private static abstract class DataApiManagerMock implements DataApiManager {
        private ListF<DataRecord> records = Cf.list();

        public void init(int numOfRecords) {
            records = Cf.range(0, numOfRecords).map(i -> new DataRecord(
                    UID, new DataRecordId(null, "test", "record_" + i), 1, Cf.map()));
        }

        public int getRecordsCount(UserDatabaseSpec databaseSpec, RecordsFilter filter) {
            return getRecords(databaseSpec, filter).size();
        }

        public ListF<DataRecord> getRecords(UserDatabaseSpec databaseSpec, RecordsFilter filter) {
            SqlLimits limits = filter.limits();
            if (limits.isAll()) {
                return records;
            } else {
                int toIndex = limits.getCount() == Integer.MAX_VALUE ?
                        records.size() :
                        Math.min(limits.getFirst() + limits.getCount(), records.size());
                int fromIndex = Math.min(limits.getFirst(), records.size());
                return records.subList(fromIndex, toIndex);
            }
        }
    }
}
