package ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata;

import org.junit.Before;
import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.bolts.collection.Option;
import ru.yandex.chemodan.app.dataapi.api.context.DatabaseAppContext;
import ru.yandex.chemodan.app.dataapi.api.db.ref.DatabaseRef;
import ru.yandex.chemodan.app.dataapi.api.db.ref.UserDatabaseSpec;
import ru.yandex.chemodan.app.dataapi.api.deltas.Delta;
import ru.yandex.chemodan.app.dataapi.api.deltas.RevisionCheckMode;
import ru.yandex.chemodan.app.dataapi.api.user.DataApiUserId;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppDatabaseSettings;
import ru.yandex.chemodan.app.dataapi.apps.settings.AppSettingsRegistry;
import ru.yandex.chemodan.app.dataapi.core.dao.test.ActivateDataApiEmbeddedPg;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata.model.AppData;
import ru.yandex.chemodan.app.dataapi.core.generic.loader.appdata.model.AppId;
import ru.yandex.chemodan.app.dataapi.core.manager.DataApiManager;
import ru.yandex.chemodan.app.dataapi.test.DataApiTestSupport;
import ru.yandex.misc.test.Assert;

/**
 * @author Denis Bakharev
 */
@ActivateDataApiEmbeddedPg
public class DatabaseAppDataStorageTest extends DataApiTestSupport {
    @Autowired
    DataApiManager dataApiManager;

    @Autowired
    DatabaseAppDataStorage databaseAppDataStorage;

    @Autowired
    AppSettingsRegistry appDataCacheSettingsRegistry;

    @Before
    public void cleanUp() {
        appDataCacheSettingsRegistry.removeAll();
    }

    @Test
    public void getAppData() {
        test(
                new AppDatabaseSettings(TestDatabaseRefs.DB1.databaseRef).enableCaching(),
                new AppDatabaseSettings(TestDatabaseRefs.DB3.databaseRef).enableCaching()
        );
    }

    @Test
    public void wildcard() {
        test(
                new AppDatabaseSettings(APP_CONTEXT.appNameO(), Option.empty()).enableCaching(),
                new AppDatabaseSettings(TestDatabaseRefs.DB3.databaseRef)
        );
    }

    private void test(AppDatabaseSettings... settings) {
        DataApiUserId uid = createRandomCleanUserInDefaultShard();

        AppId appId = new AppId(uid, APP_CONTEXT.appNameO());

        for(TestDatabaseRefs dbRef : TestDatabaseRefs.values()) {
            dataApiManager.createDatabase(dbRef.databaseSpec(uid));
        }

        for(AppDatabaseSettings setting : settings) {
            appDataCacheSettingsRegistry.setSettings(setting);
        }

        for(TestDatabaseRefs dbRef : TestDatabaseRefs.values()) {
            dataApiManager.applyDelta(dbRef.databaseSpec(uid), 0, RevisionCheckMode.PER_RECORD, dbRef.delta);
        }

        Option<AppData> appDataO = databaseAppDataStorage.find(appId);
        Assert.sizeIs(2, appDataO.get().getDatabasesMap().keys());
    }

    private static final DatabaseAppContext APP_CONTEXT = new DatabaseAppContext("app");

    private enum TestDatabaseRefs {
        DB1("dbId1", makeInsertDelta(101, "colId1")),
        DB2("dbId2", makeInsertDelta(202, "colId2")),
        DB3("dbId3", makeInsertDelta(303, "colId3"))
        ;

        final DatabaseRef databaseRef;
        final Delta delta;

        TestDatabaseRefs(String databaseId, Delta delta) {
            this.databaseRef = APP_CONTEXT.consDbRef(databaseId);
            this.delta = delta;
        }

        UserDatabaseSpec databaseSpec(DataApiUserId uid) {
            return new UserDatabaseSpec(uid, databaseRef);
        }
    }
}
